// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves Poissons equation in 2D.

#include <dolfin.h>
#include "generated_code/Poisson.h"

using namespace dolfin;
using namespace Poisson;

class Weight: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = 1.0;
  }
};

class Source: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 25.0*exp(-(dx*dx + dy*dy)/0.02);
  }
};

class BoundarySource: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = 10.0;
  }
};

class BoundaryValue: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = x[0] + x[1];
  }
};

class DirichletBoundary: public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return x[1] < DOLFIN_EPS || x[1] > 1.0-DOLFIN_EPS;
  }
};

int main()
{
    UnitSquare mesh(30,30);
    
    BilinearForm::TrialSpace V(mesh);
    CoefficientSpace_c C(mesh);
    CoefficientSpace_f F(mesh);
    CoefficientSpace_g G(mesh);
    
    BilinearForm a(V, V);
    LinearForm L(V);
    
    Weight c;
    Source f;
    BoundarySource g;
    BoundaryValue ubc;
    
    a.c = c;
    L.f = f;
    L.g = g;
    
    DirichletBoundary boundary;
    DirichletBC bc(V, ubc, boundary);
    
    Function u(V);
    solve(a==L, u, bc);
    
    File ufile("u.pvd");
    ufile << u;
    
    //plot(u);
}

