use progvis:net;
use progvis:program;
use progvis:check;
use core:io;

// Check improvements that have not yet been checked automatically.
void checkImprovements() {
	Database db;
	checkImprovements(db, true);
}

// Check improvements periodically.
void checkImprovementsLoop() {
	Database db;
	var interval = 30 min;
	print("== Begin check of improvements ==");
	checkImprovements(db, true);
	print("== End check of improvements ==");
	print("Periodic checks will continue every " + interval.toS);

	while (true) {
		checkImprovements(db, false);
		sleep(interval);
	}
}


// Check all improvements.
void checkImprovements(Database db, Bool verbose) {
	for (problem in db.problemList()) {
		if (verbose)
			print("Checking problem ${problem.id}: ${problem.title}");
		checkImprovements(db, problem);
	}
}

// Check all improvements for a particular problem.
void checkImprovements(Database db, ProblemInfo problem) {
	Int[] impls = db.allImplementations(problem.id);
	Int[] tests = db.allTests(problem.id);
	impls << -1;
	tests << -1;
	for (impl in impls) {
		for (test in tests) {
			if (db.findError(problem.id, impl, test).unknown) {
				print("Checking ${problem.id} - ${problem.title}: impl ${impl} with test ${test}...");
				checkImprovement(db, problem.id, impl, test);
			}
		}
	}
}

// Check a single combination of test + implementation.
void checkImprovement(Database db, Int problemId, Int implId, Int testId) {
	Problem problem = db.problem(problemId, implId, testId);

	MemoryProtocol memory;
	Url test = problem.test.put("test", memory);
	Url impl = problem.impl.put("impl", memory);

	Error result = Error:unknown();
	try {
		if (error = problem.checks.check([test, impl])) {
			result = Error:error(error.error.type);
			print("-- Found error: " + error.error.type);
		} else {
			result = Error:success();
			print("-- No error!");
		}
	} catch (Exception e) {
		print("-- Failed: " + e.message);
		result = Error:error("internal error");
	}

	if (!db.addError(problemId, implId, testId, result)) {
		print("-- Failed to save error to database.");
	}
}
