/* Copyright (C) 1993,1994 by the author(s).
 
 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with ShapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
*/
/*
 * AtFS -- Attribute Filesystem
 *
 * atfs.h - Type and Constant-Definitions for the 
 *	    Attribute-Filesystem
 *
 * Author: Andreas Lampen (Andreas.Lampen@cs.tu-berlin.de)
 *
 * $Header: atfs.h[7.0] Fri Dec 17 18:11:30 1993 andy@cs.tu-berlin.de frozen $
 */

#ifndef _ATFSHDR_
#define _ATFSHDR_

/*=========================================================================
 * Installation dependent constants 
 *=========================================================================*/

#include "config.h"

#define USER_MAX 64

/*=========================================================================
 * External Type Definitions
 *=========================================================================*/

typedef struct afkey Af_key;
typedef struct afuser Af_user;

#include "afsys.h"

/**** filekey ****/
struct afkey {
		Af_revlist *af_ldes;		/* ptr to revlist descriptor */
		int        af_lpos;             /* position in revlist */
	       };

/**** Set descriptor ****/
typedef struct {
		int	af_nkeys;
		int	af_setlen;
		Af_key	*af_klist;
	       } Af_set;

/**** User Identification ****/
struct afuser {
                char	af_username[USER_MAX];
		char	af_userhost[HOST_MAX];
		char	af_userdomain[DOMAIN_MAX];
	       };

/**** Attribute buffer ****/

#define AF_MAXUDAS 255

typedef struct {
		char	af_host[HOST_MAX]; 	/* hostname */
		char	af_syspath[PATH_MAX];   /* system path */
		char	af_name[NAME_MAX+1];	/* filename */
		char	af_type[TYPE_MAX];	/* filename extension (type) */
		int	af_gen;			/* generation number */
		int	af_rev;			/* revision number */
		int	af_state;		/* version state (see below) */
		Af_user af_owner;		/* owner */
		Af_user af_author;		/* author */
		size_t  af_size;                /* size of file */
		mode_t  af_mode;		/* protection (from inode) */
		Af_user af_locker;              /* locker */
		time_t	af_mtime;		/* date of last modification */
		time_t	af_atime;		/* date of last access */
		time_t	af_ctime;		/* date of last status change*/
		time_t	af_stime;		/* save date */
		time_t  af_ltime;               /* date of last lock change */
		char	*af_udattrs[AF_MAXUDAS+1]; /* user def. attributes */
	       } Af_attrs;

/*=========================================================================
 * Global Variables and Constant Definitions
 *=========================================================================*/

/**** Version states ****/

#define AF_DERIVED     -2
#define AF_NOSTATE     -1
#define	AF_BUSY		0
#define AF_SAVED	1
#define AF_PROPOSED	2
#define AF_PUBLISHED	3
#define AF_ACCESSED	4
#define AF_FROZEN	5

/**** Settings for Attrbuf ****/

#define AF_NONAME       "\0"
#define AF_NOTYPE       "\0"
#define AF_NOVNUM	-1
#define AF_NOSIZE	-1
#define AF_NOMODE	0
#define AF_NOUSER	{ "\01$\0", "\0", "\0" }
#define AF_NOTIME       -1

/**** Modes ****/

#define AF_ADD          1   /* user-defined attributes */
#define AF_REMOVE       2   /* " */
#define AF_REPLACE      3   /* " */

#define AF_BUSYVERS     -2   /* version identification */
#define AF_FIRSTVERS    -3   /* " */
#define AF_LASTVERS     -4   /* " */

#define AF_STORE_DELTA		0   /* version storage type */
#define AF_STORE_COMPLETE	1   /* " */
#define AF_STORE_COMPRESSED	2   /* " */

#define AF_LOGICAL_SUCC  1   /* logical successor */
#define AF_LOGICAL_PRED  2   /* logical predecessor */
#define AF_PHYSICAL_SUCC 3   /* physical successor */
#define AF_PHYSICAL_PRED 4   /* physical predecessor */

#define AF_LASTPOS      -4   /* set position */

/**** Object classes ****/

#define AF_CLASS_SOURCE  00
#define AF_CLASS_DERIVED 01
#define AF_VALID         02

/*=========================================================================
 * Attribute names
 *=========================================================================*/

#define AF_UDANAMLEN    128       /* maximum length of uda name */
#define AF_ATTNUM       19        /* number of attrributes in attrbuf */

#define AF_ATTHOST	"af_host"
#define AF_ATTSPATH	"af_syspath"
#define AF_ATTNAME      "af_name"
#define AF_ATTTYPE      "af_type"
#define AF_ATTUNIXNAME  "af_unixname" /* unix file name -- not for sorting */
#define AF_ATTUNIXPATH  "af_unixpath" /* full unix path name -- not for sorting */
#define AF_ATTBOUND	"af_bound"   /* bound notation (e.g. foo.c[1.2]) */
#define AF_ATTBOUNDPATH	"af_boundpath" /* " plus path (e.g. /u/andy/foo.c[1.2]) -- not for sorting */
#define AF_ATTGEN	"af_gen"
#define AF_ATTREV	"af_rev"
#define AF_ATTVERSION	"af_version" /* compound attribute: gen.rev */
#define AF_ATTSTATE	"af_state"
#define AF_ATTOWNER	"af_owner"
#define AF_ATTAUTHOR	"af_author"
#define AF_ATTDSIZE     "af_dsize"   /* not for sorting */
#define AF_ATTSIZE      "af_size"
#define AF_ATTMODE      "af_mode"
#define AF_ATTLOCKER    "af_locker"
#define AF_ATTMTIME	"af_mtime"
#define AF_ATTATIME	"af_atime"
#define AF_ATTCTIME	"af_ctime"
#define AF_ATTSTIME	"af_stime"
#define AF_ATTLTIME	"af_ltime"



/*=========================================================================
 * Errorcodes
 *=========================================================================*/

/* General */
#define AF_OK		 0

#define AF_ESYSERR	 -2	/* Error during execution of Syslib-command */
#define AF_EACCES        -3     /* permission denied */
#define AF_EARCHANGED    -4     /* archive has changed since last read */
#define AF_EBUSY         -7     /* spec. revision must not be a busy version */
#define AF_EDERIVED      -8     /* spec. revision is a derived object */
#define AF_EINVKEY 	 -9	/* invalid key */
#define AF_EINVSET	-10 	/* invalid set */
#define AF_EINVUSER     -11     /* invalid user */
#define AF_EINVVNUM     -12     /* bad version number */
#define AF_EMISC        -15     /* miscellaneous errors */
#define AF_EMODE        -16     /* invalid mode */
#define AF_ENOATFSDIR   -17     /* no AtFS subdirectory */
#define AF_ENOKEY       -18     /* key does not exist in set */
#define AF_ENOPOS       -19     /* invalid position in set */
#define AF_ENOREV       -20     /* specified revision does not exist */
#define AF_ENOTBUSY     -21     /* specified object is no busy version */
#define AF_ENOTDERIVED  -22     /* specified object is no derived object */
#define AF_ENOTLOCKED   -23     /* version is not locked or - by another user*/
#define AF_ENOTREGULAR  -24     /* specified object is no regular file */
#define AF_ENOTVERS     -25     /* specified object has no versions */
#define AF_ENOUDA       -26     /* user defined attribute does not exist */
#define AF_ESAVED       -27     /* saved versions cannot be modified */
#define AF_ESTATE	-28	/* invalid state transition */
#define AF_EWRONGSTATE  -31     /* wrong state */

/* codes for really serious errors */
#define AF_EDELTA       -32     /* error during delta operation */
#define AF_EINCONSIST	-33	/* Archive file inconsistent */
#define AF_EINTERNAL	-34	/* internal error */
#define AF_ENOATFSFILE  -35     /* no AtFS file */

/*=========================================================================
 * Declarations (ordered alphabetically)
 *=========================================================================*/

#ifdef __STDC__
#define A(alist) alist
#else
#define A(alist) ()
#endif

extern int af_noReadLock;	/* switch off archive file reader/writer locking */
extern int af_suppressUdas;	/* do not read user defined attributes from archive file */

int	 af_abort	A((void));
int	 af_access	A((char *path, char *name, char *type, int mode));
char*	 af_afname	A((char *unixname));
char*	 af_afpath	A((char *unixname));
char*	 af_aftype	A((char *unixname));
Af_user* af_afuser	A((uid_t uid));
int	 af_allattrs	A((Af_key *key, Af_attrs *attrbuf));
int	 af_cachefind	A((Af_attrs *attrbuf, Af_set *set));
char**	 af_cachenames	A((char *path, char *pattern));
int	 af_cachesize	A((char *path, int tMax, int nMax, int aMax, int *otMax, int *onMax, int *oaMax));
int	 af_chauthor	A((Af_key *key, Af_user *author));
int	 af_chmod	A((Af_key *key, int mode));
int	 af_chowner	A((Af_key *key, Af_user *owner));
void	 af_cleanup	A((void));
int	 af_close	A((FILE *file));
int	 af_commit	A((void));
int	 af_copyset	A((Af_set *set1, Af_set *set2));
int	 af_crkey	A((char *path, char *name, char *type, Af_key *key));
int	 af_diff	A((Af_set *set1, Af_set *set2, Af_set *newset));
int	 af_dropall	A((void));
int	 af_dropkey	A((Af_key *key));
int	 af_dropset	A((Af_set *set));
char*	 af_errmsg	A((char *string));
int	 af_establish	A((Af_key *key, char *fileName));
int	 af_find	A((Af_attrs *attrbuf, Af_set *set));
void	 af_freeattr	A((char *attrValue));
void	 af_freeattrbuf	A((Af_attrs *attrBuf));
int	 af_getkey	A((char *syspath, char *name, char *type, int gen, int rev, Af_key *key));
char**	 af_histories	A((char *path, char *pattern));
void	 af_initattrs	A((Af_attrs *attrs));
void	 af_initset	A((Af_set *set));
int	 af_intersect	A((Af_set *set1, Af_set *set2, Af_set *newset));
int	 af_isstdval	A((char *attrValue));
Af_user* af_lock	A((Af_key *key, Af_user *locker));
int	 af_newgen	A((Af_key *key, Af_key *newkey));
int	 af_nrofkeys	A((Af_set *set));
FILE*	 af_open	A((Af_key *key, char *mode));
void	 af_perror	A((char *string));
int	 af_predsucc	A((Af_key *inkey, int mode, Af_key *outkey));
int	 af_restore	A((Af_key *key, Af_key *restkey));
char *	 af_retattr	A((Af_key *key, char *attr));
int	 af_retnumattr	A((Af_key *key, char *attr));
time_t	 af_rettimeattr	A((Af_key *key, char *attr));
Af_user* af_retuserattr	A((Af_key *key, char *attr));
int	 af_rm		A((Af_key *key));
char*	 af_rnote	A((Af_key *key));
int	 af_savecache	A((Af_key *busykey, Af_key *savekey, char *uniqAttr, int mode));
int	 af_saverev	A((Af_key *busykey, Af_key *savekey, int gen, int storetype));
int	 af_setaddkey	A((Af_set *set, int pos, Af_key *key));
char*	 af_setarchpath	A((char *pathname));
int	 af_setattr	A((Af_key *key, int mode, char *attr));
int	 af_setbusy	A((Af_key *busykey, Af_key *key, Af_key *prevkey));
int	 af_setgkey	A((Af_set *set, int pos, Af_key *key));
int	 af_setposrmkey	A((Af_set *set, int pos));
int	 af_setrmkey	A((Af_set *set, Af_key *key));
int	 af_snote	A((Af_key *key, char *buf));
int	 af_sortset	A((Af_set *set, char *attr));
int	 af_sstate	A((Af_key *key, int state));
int	 af_subset	A((Af_set *set, Af_attrs *attrbuf, Af_set *newset));
int	 af_svnum	A((Af_key *key, int gen, int rev));
Af_user* af_testlock	A((Af_key *key));
void	 af_transaction	A((void));
int	 af_union	A((Af_set *set1, Af_set *set2, Af_set *newset, int checkDouble));
Af_user* af_unlock	A((Af_key *key));
char*	 af_version	A((void));

#endif /* __ATFSHDR__ */

