#!/bin/bash
# Copyright (c) 2017 - 2021 Advanced Micro Devices, Inc. All Rights Reserved.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

# Parse command-line options
# Option strings
SHORT=h
LONG=help,opencl:,hip:,rocclr:
# read the options
OPTS=$(getopt --options $SHORT --long $LONG --name "$0" -- "$@")
if [ $? != 0 ] ; then echo "Failed to parse options...exiting." >&2 ; exit 1 ; fi

usage() {
    echo "Usage: $0 --hip <PATH to the hip common src> --opencl <PATH to the opencl src> --rocclr <PATH to the rocclr src>" ;
    exit 1;
}

[ $# -eq 0 ] && usage

eval set -- "$OPTS"

# extract options and their arguments into variables.
while true ; do
  case "$1" in
    --hip )
      HIP_DIR="$2"
      shift 2
      ;;
    --rocclr )
      ROCCLR_DIR="$2"
      shift 2
      ;;
    --opencl )
      OPENCL_DIR="$2"
      shift 2
      ;;
    -h | --help )
      usage
      shift
      ;;
    -- )
      shift
      break
      ;;
    *)
      echo "Internal error!"
      exit 1
      ;;
  esac
done

BUILD_ROOT="$( mktemp -d )"
SRC_ROOT="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
WORKING_DIR=$PWD
DASH_JAY="-j $(getconf _NPROCESSORS_ONLN)"
OS_NAME="$(cat /etc/os-release | awk -F '=' '/^NAME/{print $2}' | awk '{print $1}' | tr -d '"')"
[[ -z "$ROCM_PATH" ]] && ROCM_PATH=/opt/rocm

err() {
    echo "${1-Died}." >&2
}

die() {
    err "$1"
    exit 1
}

pushd () {
    command pushd "$@" > /dev/null
}

popd () {
    command popd "$@" > /dev/null
}

function setupENV()
{
    if [ "$OS_NAME" == "Ubuntu" ]
    then
      sudo apt-get update
      sudo apt-get install dpkg-dev rpm doxygen libelf-dev rename liburi-encode-perl \
         libfile-basedir-perl libfile-copy-recursive-perl libfile-listing-perl
    elif [ "$OS_NAME" == "CentOS" ]
    then
      yum install dpkg-dev rpm-build doxygen elfutils-libelf-devel prename \
         perl-URI-Encode perl-File-Listing perl-File-BaseDir
    fi
}

function buildHIP()
{
    pushd $BUILD_ROOT
    HIP_BUILD_DIR="$BUILD_ROOT/hip_build"
    mkdir $HIP_BUILD_DIR
    pushd $HIP_BUILD_DIR
    cmake $SRC_ROOT -DHIP_COMMON_DIR="$HIP_DIR" -DAMD_OPENCL_PATH=$OPENCL_DIR -DROCCLR_PATH=$ROCCLR_DIR -DCMAKE_PREFIX_PATH="$ROCM_PATH" -DCMAKE_BUILD_TYPE=Release
    make $DASH_JAY
    make package
    if [ "$OS_NAME" == "Ubuntu" ]
    then
      cp hip-*.deb $WORKING_DIR
      sudo dpkg -i -B hip-dev*.deb hip-runtime-amd*.deb hip-sample*.deb hip-doc*.deb
    elif [ "$OS_NAME" == "CentOS" ]
    then
      cp hip-*.rpm $WORKING_DIR
      sudo rpm -ivh --replacefiles --force hip-devel*.rpm hip-runtime-amd*.rpm hip-sample*.rpm \
         hip-doc*.rpm
    fi
    popd
    popd
    rm -rf $BUILD_ROOT
}

echo "Preparing build environment"
setupENV || die "setupENV failed"
echo "Building and installing HIP packages"
buildHIP || die "buildHIP failed"
echo "Finished building HIP packages"
