# ifndef _RHEOLEF_FIELD_INDIRECT_H
# define _RHEOLEF_FIELD_INDIRECT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/field.h"

namespace rheolef {

// forward:
template <class T, class M>
class field_indirect_const;

// =================================================================================
// field_indirect 
// as for:
//   uh["left"] = value
//   uh["left"] = gh
// =================================================================================
template <class T, class M = rheo_default_memory_model>
class field_indirect {
public:
//protected:

// typedefs

  typedef typename field_basic<T,M>::size_type size_type;
  typedef T                                    scalar_type;
  typedef T                                    value_type; // TO_CLEAN !
  class iterator;
  class const_iterator;

// allocators:

  field_indirect (field_basic<T,M>& uh, const geo_basic<T,M>& dom)
    : _V(uh.get_space()),
      _W(dom, _V.get_numbering().name(), _V.valued()),
      _dom(dom),
      _indirect(_V.build_indirect_array (_W, dom)),
      _val(uh.begin_dof())
    {
    }

  field_indirect (field_component<T,M>& uh, const geo_basic<T,M>& dom);

// accessors & modifiers:
public:

  const T& operator=  (const T& alpha);
  field_indirect<T,M>& operator= (const field_basic<T,M>& expr);
  field_indirect<T,M>& operator= (const field_indirect<T,M>& expr);
  field_indirect<T,M>& operator= (const field_indirect_const<T,M>& expr);

//protected:
  const distributor& ownership() const { return _indirect.ownership(); }
  const communicator& comm() const { return ownership().comm(); }
  size_type     ndof() const { return ownership().size(); }
  size_type dis_ndof() const { return ownership().dis_size(); }
  std::string stamp() const;
  const space_basic<T,M>& get_space() const { return _W; }
        T& dof (size_type idof)       { return _val [_indirect [idof]]; }
  const T& dof (size_type idof) const { return _val [_indirect [idof]]; }

  iterator begin_dof();
  iterator end_dof();
  const_iterator begin_dof() const;
  const_iterator end_dof() const;
protected:
  friend class field_basic<T,M>;
  friend class field_indirect_const<T,M>;
// data:
  space_basic<T,M>                      _V;
  space_basic<T,M>                      _W;
  geo_basic<T,M>                        _dom;
  array<size_type,M>                    _indirect; // temporary
  typename field_basic<T,M>::iterator   _val;      // iterator = reference on an external variable uh
						   // => cstor is protected : prevent copy outside the field class
};
template<class T, class M>
inline
const T&
field_indirect<T,M>::operator=  (const T& alpha)
{
  std::fill (begin_dof(), end_dof(), alpha);
  return alpha;
}
template<class T, class M>
inline
field_indirect<T,M>&
field_indirect<T,M>::operator=  (const field_basic<T,M>& expr)
{
  check_macro (_V.valued_tag() == space_constant::scalar, "field[domain]: unsupported non-scalar field");
  check_macro (stamp() == expr.stamp(), "field[domain] = field : incompatible spaces "
		<< stamp() << " and " << expr.stamp());
  algo::copy_n (expr.begin_dof(), expr.ndof(), begin_dof());
  return *this;
}
template<class T, class M>
inline
field_indirect<T,M>&
field_indirect<T,M>::operator=  (const field_indirect<T,M>& expr)
{
  check_macro (_V.valued_tag() == space_constant::scalar, "field[domain]: unsupported non-scalar field");
  check_macro (stamp() == expr.stamp(), "field[domain] = field[domain]: incompatible spaces "
		<< stamp() << " and " << expr.stamp());
  algo::copy_n (expr.begin_dof(), expr.ndof(), begin_dof());
  return *this;
}
template<class T, class M>
inline
std::string
field_indirect<T,M>::stamp() const
{
  // e.g. "P1(square.left)", an unique signature for field_expr<Expr> size-like checks
  return _V.get_numbering().name() + "(" + _V.get_geo().name() + "[" + _dom.name() + "])";
}
template<class T, class M>
inline
field_indirect<T,M>
field_basic<T,M>::operator[] (const geo_basic<T,M>& dom)
{
  dis_dof_update_needed();
  return field_indirect<T,M> (*this, dom);
}
template<class T, class M>
inline
field_indirect<T,M>
field_basic<T,M>::operator[] (std::string dom_name)
{
  dis_dof_update_needed();
  return operator[] (get_space().get_geo().operator[] (dom_name));
}
template<class T, class M>
inline
field_basic<T,M>&
field_basic<T,M>::operator=  (const field_indirect<T,M>& expr)
{
  if (stamp() == "") {
    resize (expr.get_space());
  } else {
    check_macro (stamp() == expr.stamp(), "incompatible spaces "
	<< stamp() << " and " << expr.stamp()
	<< " in field = field[domain]");
  }
  dis_dof_update_needed();
  algo::copy_n (expr.begin_dof(), expr.ndof(), begin_dof());
  return *this;
}
template<class T, class M>
inline
field_basic<T,M>::field_basic (const field_indirect<T,M>& expr)
 : _V(),
   _u(),
   _b(),
   _dis_dof_update_needed(true)
{
    operator= (expr);
}
// =================================================================================
// field_indirect::iterator
// =================================================================================
template <class T, class M>
class field_indirect<T,M>::iterator {
public:
// typedef:
  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef T&                                reference;
  typedef T*                                pointer;
  typedef std::ptrdiff_t                    difference_type;
// allocator:
  iterator (typename array<size_type,M>::const_iterator idof_iter, typename field_basic<T,M>::iterator val)
    : _idof_iter(idof_iter), _val(val) {}
 
// accessors & modifiers:
        T& operator* ()       { return _val [*_idof_iter]; }
  const T& operator* () const { return _val [*_idof_iter]; }
  iterator& operator++()      { ++_idof_iter; return *this; }

// comparators:

  bool operator== (const iterator& j) const { return _idof_iter == j._idof_iter; }
  bool operator!= (const iterator& j) const { return ! operator== (j); }
protected:
// data:
  typename array<size_type,M>::const_iterator _idof_iter;
  typename field_basic<T,M>::iterator         _val;
};
template<class T, class M>
inline
typename field_indirect<T,M>::iterator
field_indirect<T,M>::begin_dof()
{
  return iterator (_indirect.begin(), _val);
}
template<class T, class M>
inline
typename field_indirect<T,M>::iterator
field_indirect<T,M>::end_dof()
{
  return iterator (_indirect.end(), _val);
}
// =================================================================================
// field_indirect::const_iterator
// =================================================================================
template <class T, class M>
class field_indirect<T,M>::const_iterator {
public:
// typedef:
  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef const T&                          reference;
  typedef const T*                          pointer;
  typedef std::ptrdiff_t                    difference_type;
// allocator:
  const_iterator (typename array<size_type,M>::const_iterator idof_iter, typename field_basic<T,M>::const_iterator val)
    : _idof_iter(idof_iter), _val(val) {}
 
// accessors & modifiers:
  const T& operator* () const   { return _val [*_idof_iter]; }
  const_iterator& operator++()  { ++_idof_iter; return *this; }

// comparators:

  bool operator== (const const_iterator& j) const { return _idof_iter == j._idof_iter; }
  bool operator!= (const const_iterator& j) const { return ! operator== (j); }
protected:
// data:
  typename array<size_type,M>::const_iterator _idof_iter;
  typename field_basic<T,M>::const_iterator   _val;
};
template<class T, class M>
inline
typename field_indirect<T,M>::const_iterator
field_indirect<T,M>::begin_dof() const
{
  return const_iterator (_indirect.begin(), _val);
}
template<class T, class M>
inline
typename field_indirect<T,M>::const_iterator
field_indirect<T,M>::end_dof() const
{
  return const_iterator (_indirect.end(), _val);
}
// =================================================================================
// field_indirect_const 
// as for:
//   gh = uh["left"]
// =================================================================================
template <class T, class M = rheo_default_memory_model>
class field_indirect_const {
public:
//protected:

// typedefs:

  typedef typename field_basic<T,M>::size_type size_type;
  typedef T                                    scalar_type;
  typedef T                                    value_type; // TO_CLEAN !
  class const_iterator;

// allocators:

  field_indirect_const (const field_basic<T,M>& uh, const geo_basic<T,M>& dom)
    : _V(uh.get_space()),
      _W(dom, _V.get_numbering().name(), _V.valued()),
      _dom(dom),
      _indirect(_V.build_indirect_array (_W, dom)),
      _val(uh.begin_dof())
    {
    }

  field_indirect_const (field_indirect<T,M> gh)
    : _V(gh._V),
      _W(gh._W),
      _dom(gh._dom),
      _indirect(gh._indirect),
      _val(gh._val)
    {}

// accessors:
//protected:

  const distributor& ownership() const { return _indirect.ownership(); }
  const communicator& comm() const { return ownership().comm(); }
  size_type     ndof() const { return ownership().size(); }
  size_type dis_ndof() const { return ownership().dis_size(); }
  std::string stamp() const;
  space_basic<T,M> get_space() const { return _W; }

  const T& dof(size_type idof) const { return _val [_indirect [idof]]; }

  const_iterator begin_dof() const;
  const_iterator end_dof() const;
  friend class field_basic<T,M>;
protected:
  space_basic<T,M>                            _V;
  space_basic<T,M>                            _W;
  geo_basic<T,M>                              _dom;
  array<size_type,M>                          _indirect; // temporary
  typename field_basic<T,M>::const_iterator   _val;      // iterator = reference on an external variable uh
				   		         // thus cstor is protected to prevent copy outside the field class
};
template<class T, class M>
inline
std::string
field_indirect_const<T,M>::stamp() const
{
  // e.g. "P1(square.left)", an unique signature for field_expr<Expr> size-like checks
  return _V.get_numbering().name() + "(" + _V.get_geo().name() + "[" + _dom.name() + "])";
}
template<class T, class M>
inline
field_indirect_const<T,M>
field_basic<T,M>::operator[] (const geo_basic<T,M>& dom) const
{
  return field_indirect_const<T,M> (*this, dom);
}
template<class T, class M>
inline
field_indirect_const<T,M>
field_basic<T,M>::operator[] (std::string dom_name) const
{
  return operator[] (get_space().get_geo().operator[] (dom_name));
}
template<class T, class M>
inline
field_basic<T,M>::field_basic (const field_indirect_const<T,M>& expr)
 : _V(),
   _u(),
   _b(),
   _dis_dof_update_needed(true)
{
    operator= (expr);
}
template<class T, class M>
inline
field_basic<T,M>&
field_basic<T,M>::operator=  (const field_indirect_const<T,M>& expr)
{
  if (stamp() == "") {
    resize (expr.get_space());
  } else {
    check_macro (stamp() == expr.stamp(), "incompatible spaces "
	<< stamp() << " and " << expr.stamp()
	<< " in field = field[domain]");
  }
  dis_dof_update_needed();
  algo::copy_n (expr.begin_dof(), expr.ndof(), begin_dof());
  return *this;
}
template<class T, class M>
inline
field_indirect<T,M>&
field_indirect<T,M>::operator=  (const field_indirect_const<T,M>& expr)
{
  check_macro (_V.valued_tag() == space_constant::scalar, "field[domain]: unsupported non-scalar field");
  check_macro (stamp() == expr.stamp(), "field[domain] = field[domain]: incompatible spaces "
		<< stamp() << " and " << expr.stamp());
  algo::copy_n (expr.begin_dof(), expr.ndof(), begin_dof());
  return *this;
}
// =================================================================================
// field_indirect_const::const_iterator
// =================================================================================
template <class T, class M>
class field_indirect_const<T,M>::const_iterator {
public:
// typedefs:

  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef const T&                          reference;
  typedef const T*                          pointer;
  typedef std::ptrdiff_t                    difference_type;

// allocator:
  const_iterator (typename array<size_type,M>::const_iterator idof_iter, typename field_basic<T,M>::const_iterator val)
    : _idof_iter(idof_iter), _val(val) {}
 
// accessors & modifiers:
  const T& operator* () const { return _val [*_idof_iter]; }
  const_iterator& operator++()      { ++_idof_iter; return *this; }

// comparators:

  bool operator== (const const_iterator& j) const { return _idof_iter == j._idof_iter; }
  bool operator!= (const const_iterator& j) const { return ! operator== (j); }
protected:
// data:
  typename array<size_type,M>::const_iterator _idof_iter;
  typename field_basic<T,M>::const_iterator   _val;
};
template<class T, class M>
inline
typename field_indirect_const<T,M>::const_iterator
field_indirect_const<T,M>::begin_dof() const
{
  return const_iterator (_indirect.begin(), _val);
}
template<class T, class M>
inline
typename field_indirect_const<T,M>::const_iterator
field_indirect_const<T,M>::end_dof() const
{
  return const_iterator (_indirect.end(), _val);
}
// =================================================================================
// io
// =================================================================================
template <class T, class M>
inline
odiststream&
operator << (odiststream& ops, const field_indirect<T,M>& uh)
{
    field_basic<T,M> tmp = uh;
    return tmp.put (ops);
}
template <class T, class M>
inline
odiststream&
operator << (odiststream& ops, const field_indirect_const<T,M>& uh)
{
    field_basic<T,M> tmp = uh;
    return tmp.put (ops);
}

}// namespace rheolef
# endif /* _RHEOLEF_FIELD_INDIRECT_H */
