% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clanistic.R
\name{getClans}
\alias{getClans}
\alias{getSlices}
\alias{getClips}
\alias{getDiversity}
\alias{summary.clanistics}
\alias{diversity}
\title{Clans, slices and clips}
\usage{
getClans(tree)

getSlices(tree)

getClips(tree, all = TRUE)

getDiversity(tree, x, norm = TRUE, var.names = NULL, labels = "new")

\method{summary}{clanistics}(object, ...)

diversity(tree, X)
}
\arguments{
\item{tree}{An object of class phylo or multiPhylo (getDiversity).}

\item{all}{A logical, return all or just the largest clip.}

\item{x}{An object of class phyDat.}

\item{norm}{A logical, return Equitability Index (default) or Shannon
Diversity.}

\item{var.names}{A vector of variable names.}

\item{labels}{see details.}

\item{object}{an object for which a summary is desired.}

\item{...}{Further arguments passed to or from other methods.}

\item{X}{a data.frame}
}
\value{
getClans, getSlices and getClips return a matrix of partitions, a
matrix of ones and zeros where rows correspond to a clan, slice or clip and
columns to tips. A one indicates that a tip belongs to a certain partition.
\cr getDiversity returns a list with tree object, the first is a data.frame
of the equitability index or Shannon divergence and parsimony scores
(p-score) for all trees and variables. The data.frame has two attributes,
the first is a splits object to identify the taxa of each tree and the
second is a splits object containing all partitions that perfectly fit.
}
\description{
Functions for clanistics to compute clans, slices, clips for unrooted trees
and functions to quantify the fragmentation of trees.
}
\details{
Every split in an unrooted tree defines two complementary clans. Thus for an
unrooted binary tree with \eqn{n} leaves there are \eqn{2n - 3} edges, and
therefore \eqn{4n - 6} clans (including \eqn{n} trivial clans containing
only one leave).

Slices are defined by a pair of splits or tripartitions, which are not
clans. The number of distinguishable slices for a binary tree with \eqn{n}
tips is \eqn{2n^2 - 10n + 12}.

%A clip is a different type of partition as it is defined by evolutionary or
cophenetic distance and not by the topology. Namely clips are groups of
leaves for which the maximum pairwise distance is smaller than threshold.
%For a better separation we additionally demand that the maximum pairwise
distance within a clip is lower than the distance between any member of the
clip and any other tip.

A clip is a different type of partition, defining groups of leaves that are
related in terms of evolutionary distances and not only topology.  Namely,
clips are groups of leaves for which all pairwise path-length distances are
smaller than a given threshold value (Lapointe et al. 2010). There exists
different numbers of clips for different thresholds, the largest (and
trivial) one being the whole tree.  There is always a clip containing only
the two leaves with the smallest pairwise distance.

Clans, slices and clips can be used to characterize how well a vector of
categorial characters (natives/intruders) fit on a tree. We will follow the
definitions of Lapointe et al.(2010).  A complete clan is a clan that
contains all leaves of a given state/color, but can also contain leaves of
another state/color. A clan is homogeneous if it only contains leaves of one
state/color.

\code{getDiversity} computes either the \cr Shannon Diversity: \eqn{H =
-\sum_{i=1}^{k}(N_i/N) log(N_i/N), N=\sum_{i=1}^{k} N_i}{H = -sum(N_i/N) *
log(N_i/N), N=sum(N_i)} \cr or the \cr Equitability Index: \eqn{E = H /
log(N)} \cr where \eqn{N_i} are the sizes of the \eqn{k} largest homogeneous
clans of intruders.  If the categories of the data can be separated by an
edge of the tree then the E-value will be zero, and maximum equitability
(E=1) is reached if all intruders are in separate clans. getDiversity
computes these Intruder indices for the whole tree, complete clans and
complete slices. Additionally the parsimony scores (p-scores) are reported.
The p-score indicates if the leaves contain only one color (p-score=0), if
the the leaves can be separated by a single split (perfect clan, p-score=1)
or by a pair of splits (perfect slice, p-score=2).

So far only 2 states are supported (native, intruder), however it is also
possible to recode several states into the native or intruder state using
contrasts, for details see section 2 in vignette("phangorn-specials").
Furthermore unknown character states are coded as ambiguous character, which
can act either as native or intruder minimizing the number of clans or
changes (in parsimony analysis) needed to describe a tree for given data.

Set attribute labels to "old" for analysis as in Schliep et al. (2010) or to
"new" for names which are more intuitive.

\code{diversity} returns a data.frame with the parsimony score for each tree
and each levels of the variables in \code{X}. \code{X} has to be a
\code{data.frame} where each column is a factor and the rownames of \code{X}
correspond to the tips of the trees.

%TODO See also vignette("Clanistic").
}
\examples{

set.seed(111)
tree <- rtree(10)
getClans(tree)
getClips(tree, all=TRUE)
getSlices(tree)

set.seed(123)
trees <- rmtree(10, 20)
X <- matrix(sample(c("red", "blue", "violet"), 100, TRUE, c(.5,.4, .1)),
   ncol=5, dimnames=list(paste('t',1:20, sep=""), paste('Var',1:5, sep="_")))
x <- phyDat(X, type = "USER", levels = c("red", "blue"), ambiguity="violet")
plot(trees[[1]], "u", tip.color = X[trees[[1]]$tip,1])  # intruders are blue

(divTab <- getDiversity(trees, x, var.names=colnames(X)))
summary(divTab)

}
\references{
Lapointe, F.-J., Lopez, P., Boucher, Y., Koenig, J., Bapteste,
E. (2010) Clanistics: a multi-level perspective for harvesting unrooted gene
trees. \emph{Trends in Microbiology} 18: 341-347

Wilkinson, M., McInerney, J.O., Hirt, R.P., Foster, P.G., Embley, T.M.
(2007) Of clades and clans: terms for phylogenetic relationships in unrooted
trees. \emph{Trends in Ecology and Evolution} 22: 114-115

Schliep, K., Lopez, P., Lapointe F.-J., Bapteste E. (2011) Harvesting
Evolutionary Signals in a Forest of Prokaryotic Gene Trees, \emph{Molecular
Biology and Evolution} 28(4): 1393-1405
}
\seealso{
\code{\link{parsimony}}, Consistency index \code{\link{CI}},
Retention index \code{\link{RI}}, \code{\link{phyDat}}
}
\author{
Klaus Schliep \email{klaus.schliep@snv.jussieu.fr}

Francois-Joseph Lapointe \email{francois-joseph.lapointe@umontreal.ca}
}
\keyword{cluster}
