// --------------------------------------------------------------------------
//                   OpenMS -- Open-Source Mass Spectrometry               
// --------------------------------------------------------------------------
// Copyright The OpenMS Team -- Eberhard Karls University Tuebingen,
// ETH Zurich, and Freie Universitaet Berlin 2002-2018.
// 
// This software is released under a three-clause BSD license:
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  * Neither the name of any author or any participating institution 
//    may be used to endorse or promote products derived from this software 
//    without specific prior written permission.
// For a full list of authors, refer to the file AUTHORS. 
// --------------------------------------------------------------------------
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL ANY OF THE AUTHORS OR THE CONTRIBUTING 
// INSTITUTIONS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
// OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
// OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
// ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// --------------------------------------------------------------------------
// $Maintainer: Timo Sachsenberg$
// $Authors: Stephan Aiche$
// --------------------------------------------------------------------------

#include <OpenMS/CONCEPT/ClassTest.h>
#include <OpenMS/test_config.h>

///////////////////////////
#include <OpenMS/ANALYSIS/QUANTITATION/ItraqFourPlexQuantitationMethod.h>
///////////////////////////

#include <OpenMS/DATASTRUCTURES/Matrix.h>

using namespace OpenMS;
using namespace std;

START_TEST(ItraqFourPlexQuantitationMethod, "$Id$")

/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////

ItraqFourPlexQuantitationMethod* ptr = nullptr;
ItraqFourPlexQuantitationMethod* null_ptr = nullptr;
START_SECTION(ItraqFourPlexQuantitationMethod())
{
	ptr = new ItraqFourPlexQuantitationMethod();
	TEST_NOT_EQUAL(ptr, null_ptr)
}
END_SECTION

START_SECTION(~ItraqFourPlexQuantitationMethod())
{
	delete ptr;
}
END_SECTION

START_SECTION((const String& getName() const ))
{
  ItraqFourPlexQuantitationMethod quant_meth;
  TEST_STRING_EQUAL(quant_meth.getName(), "itraq4plex") 
}
END_SECTION
    
START_SECTION((ItraqFourPlexQuantitationMethod(const ItraqFourPlexQuantitationMethod &other)))
{
  ItraqFourPlexQuantitationMethod qm;
  Param p = qm.getParameters();
  p.setValue("channel_114_description", "new_description");
  p.setValue("reference_channel", 116);
  qm.setParameters(p);
  
  ItraqFourPlexQuantitationMethod qm2(qm);
  IsobaricQuantitationMethod::IsobaricChannelList channel_list = qm2.getChannelInformation();
  TEST_STRING_EQUAL(channel_list[0].description, "new_description")
  TEST_EQUAL(qm2.getReferenceChannel(), 2)    
}
END_SECTION

START_SECTION((ItraqFourPlexQuantitationMethod& operator=(const ItraqFourPlexQuantitationMethod &rhs)))
{
  ItraqFourPlexQuantitationMethod qm;
  Param p = qm.getParameters();
  p.setValue("channel_114_description", "new_description");
  p.setValue("reference_channel", 116);
  qm.setParameters(p);
  
  ItraqFourPlexQuantitationMethod qm2;
  qm2 = qm;
  IsobaricQuantitationMethod::IsobaricChannelList channel_list = qm2.getChannelInformation();
  TEST_STRING_EQUAL(channel_list[0].description, "new_description")
  TEST_EQUAL(qm2.getReferenceChannel(), 2)
}
END_SECTION


START_SECTION((const IsobaricChannelList& getChannelInformation() const ))
{
  ItraqFourPlexQuantitationMethod quant_meth;
  IsobaricQuantitationMethod::IsobaricChannelList channel_list = quant_meth.getChannelInformation();
  
  TEST_EQUAL(channel_list.size(), 4)
  ABORT_IF(channel_list.size() != 4)
  
  // descriptions are empty by default
  TEST_STRING_EQUAL(channel_list[0].description, "")
  TEST_STRING_EQUAL(channel_list[1].description, "")
  TEST_STRING_EQUAL(channel_list[2].description, "")
  TEST_STRING_EQUAL(channel_list[3].description, "")
    
  // check masses&co
  TEST_EQUAL(channel_list[0].name, 114)
  TEST_EQUAL(channel_list[0].id, 0)
  TEST_EQUAL(channel_list[0].center, 114.1112)

  TEST_EQUAL(channel_list[1].name, 115)
  TEST_EQUAL(channel_list[1].id, 1)
  TEST_EQUAL(channel_list[1].center, 115.1082)

  TEST_EQUAL(channel_list[2].name, 116)
  TEST_EQUAL(channel_list[2].id, 2)
  TEST_EQUAL(channel_list[2].center, 116.1116)

  TEST_EQUAL(channel_list[3].name, 117)
  TEST_EQUAL(channel_list[3].id, 3)
  TEST_EQUAL(channel_list[3].center, 117.1149)
}
END_SECTION

START_SECTION((Size getNumberOfChannels() const ))
{
  ItraqFourPlexQuantitationMethod quant_meth;
  TEST_EQUAL(quant_meth.getNumberOfChannels(), 4)
}
END_SECTION

START_SECTION((virtual Matrix<double> getIsotopeCorrectionMatrix() const ))
{
  ItraqFourPlexQuantitationMethod quant_meth;
  
  // we only check the default matrix here
  Matrix<double> m = quant_meth.getIsotopeCorrectionMatrix();
  TEST_EQUAL(m.rows(), 4)
  TEST_EQUAL(m.cols(), 4)
    
  ABORT_IF(m.rows() != 4)
  ABORT_IF(m.cols() != 4)

  /**  
    0.929   0.02      0      0 
    0.059  0.923   0.03  0.001 
    0.002  0.056  0.924   0.04 
        0  0.001  0.045  0.923
  */
  double real_m[4][4] = {{0.929, 0.02, 0, 0},
    {0.059, 0.923, 0.03, 0.001},
    {0.002, 0.056, 0.924, 0.04},
    {0, 0.001, 0.045, 0.923}};
  
  for(Matrix<double>::SizeType i = 0; i < m.rows(); ++i)
  {
    for(Matrix<double>::SizeType j = 0; j < m.cols(); ++j)
    {
      TEST_REAL_SIMILAR(m(i,j), real_m[i][j])
    }
  }
}
END_SECTION

START_SECTION((virtual Size getReferenceChannel() const ))
{
  ItraqFourPlexQuantitationMethod quant_meth;
  TEST_EQUAL(quant_meth.getReferenceChannel(), 0)
    
  Param p;
  p.setValue("reference_channel",115);
  quant_meth.setParameters(p);
  
  TEST_EQUAL(quant_meth.getReferenceChannel(), 1)
}
END_SECTION
  
/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////
END_TEST
