!
!  fitsgphot
!
!  Copyright © 2018 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.


module fitsgphot

  use fitsio
  use iso_fortran_env

  implicit none

  integer, parameter, private :: rp = selected_real_kind(15)

contains

  subroutine fits_gphot_read(filename,hwhm,sep,raper,xcens,ycens, &
    sky,skyerr,skycorr,skyerrcorr,apcts,apcts_err,status)

    character(len=*), intent(in) :: filename
    real, intent(out) :: hwhm, sep
    real, dimension(:), allocatable, intent(out) :: raper, xcens,ycens, &
         sky,skyerr,skycorr,skyerrcorr
    real, dimension(:,:), allocatable, intent(out) :: apcts,apcts_err
    integer, intent(in out) :: status

    integer :: blocksize,nrows,xcol,ycol,ecol,scol,i,n,naper
    integer, parameter :: extver = 0, frow = 1, felem = 1
    real, parameter :: nullval = 0.0
    logical :: anyf
    character(len=FLEN_COMMENT) :: com
    character(len=FLEN_VALUE) :: key, label

    status = 0
    call ftopen(15,filename,READONLY,blocksize,status)
    if( status /= 0 ) goto 666

    call ftmnhd(15,BINARY_TBL,APEREXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       write(error_unit,*) "Error: ",trim(APEREXTNAME)// &
            " extension not found. Has been aperture photometry by " // &
            "`munipack aphot "//trim(filename)//"' performed?"
       goto 666
    end if

    call ftgnrw(15,nrows,status)
    call ftgkyj(15,FITS_KEY_NAPER,naper,com,status)
    call ftgkye(15,FITS_KEY_HWHM,hwhm,com,status)
    call ftgkye(15,trim(FITS_KEY_ANNULUS)//'2',sep,com,status)

    allocate(xcens(nrows),ycens(nrows),sky(nrows),skyerr(nrows), &
         skycorr(nrows),skyerrcorr(nrows),apcts(nrows,naper), &
         apcts_err(nrows,naper),raper(naper))

    do i = 1, naper
       call ftkeyn(FITS_KEY_APER,i,key,status)
       call ftgkye(15,key,raper(i),com,status)
    end do

    call ftgcno(15,.true.,FITS_COL_X,xcol,status)
    call ftgcno(15,.true.,FITS_COL_Y,ycol,status)
    call ftgcno(15,.true.,FITS_COL_SKY,scol,status)
    call ftgcno(15,.true.,FITS_COL_SKYERR,ecol,status)
    call ftgcve(15,xcol,frow,felem,nrows,nullval,xcens,anyf,status)
    call ftgcve(15,ycol,frow,felem,nrows,nullval,ycens,anyf,status)
    call ftgcve(15,scol,frow,felem,nrows,nullval,sky,anyf,status)
    call ftgcve(15,ecol,frow,felem,nrows,nullval,skyerr,anyf,status)

    do i = 1, naper
       write(label,'(a,i0)') FITS_COL_APCOUNT,i
       call ftgcno(15,.true.,label,n,status)
       call ftgcve(15,n,frow,felem,nrows,nullval,apcts(:,i),anyf,status)
       write(label,'(a,i0)') FITS_COL_APCOUNTERR,i
       call ftgcno(15,.true.,label,n,status)
       call ftgcve(15,n,frow,felem,nrows,nullval,apcts_err(:,i),anyf,status)
    end do

    ! all the commands must be finished without eny error for
    ! correct head, any error indicates internal inconsistency

666 continue

    call ftclos(15,status)

    if( status /= 0 ) then
       call ftrprt('STDERR',status)
       if( allocated(xcens) ) deallocate(xcens,ycens,sky,skyerr, &
            skycorr,skyerrcorr,apcts,apcts_err,raper)
    end if


  end subroutine fits_gphot_read


  subroutine fits_find_save(filename,output,ghwhm,rflux90, &
       raper, xcens, ycens, sky,skyerr, gcount,gcount_err, growflag, &
       curve, curve_err, prof, status)

    character(len=*), intent(in) :: filename, output
    real, intent(in) :: ghwhm,rflux90
    real, dimension(:), intent(in) :: raper, xcens, ycens, sky,skyerr
    real(rp), dimension(:), intent(in) :: gcount,gcount_err
    integer, dimension(:), intent(in) :: growflag
    real(rp), dimension(:), intent(in) :: curve, curve_err, prof
    integer, intent(in out) :: status

    integer :: hdutype,n
    integer, parameter :: extver = 0, frow = 1, felem = 1, nbegin = 4
    real, parameter :: nullval = 0.0
    character(len=FLEN_VALUE), dimension(:), allocatable :: ttype, tform, tunit
    character(len=FLEN_COMMENT) :: com


    if( status /= 0 ) return

    call fits_open_file(15,filename,output,status)
    if( status /= 0 ) goto 666

    ! grow photometry table
    call ftmnhd(15,BINARY_TBL,GROWEXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       status = 0
    else
       ! already presented ? remove it !
       call ftdhdu(15,hdutype,status)
       if( status /= 0 ) goto 666
    end if

    n = 7
    allocate(ttype(n), tform(n), tunit(n))

    tform = '1D'
    tform(7) = '1B'
    tunit = ''
    ttype(1) = FITS_COL_X
    ttype(2) = FITS_COL_Y
    ttype(3) = FITS_COL_SKY
    ttype(4) = FITS_COL_SKYERR
    ttype(5) = FITS_COL_GCOUNT
    ttype(6) = FITS_COL_GCOUNTERR
    ttype(7) = FITS_COL_GROWFLAG

    call ftibin(15,0,size(ttype),ttype,tform,tunit,GROWEXTNAME,0,status)
    call ftpkye(15,FITS_KEY_HWHM,ghwhm,-4, &
         '[pix] half width at half of maximum',status)
    call ftpkye(15,FITS_KEY_RF90,rflux90,-4, &
         '[pix] radius contains 90% of flux',status)
    write(com,'(a,i0)') 'Count of stars used for curve construction: ', &
         count(growflag == 1)
    call ftpcom(15,com,status)
    call ftpcom(15,'GROWFLAG: 0 - star, not used,',status)
    call ftpcom(15,'          1 - star, used for growth curve construction,', &
         status)
    call ftpcom(15,'          2 - non-stellar object',status)
    call ftpcle(15,1,frow,felem,size(xcens),xcens,status)
    call ftpcle(15,2,frow,felem,size(ycens),ycens,status)
    call ftpcle(15,3,frow,felem,size(sky),sky,status)
    call ftpcle(15,4,frow,felem,size(skyerr),skyerr,status)
    call ftpcld(15,5,frow,felem,size(gcount),gcount,status)
    call ftpcld(15,6,frow,felem,size(gcount_err),gcount_err,status)
    call ftpclj(15,7,frow,felem,size(growflag),growflag,status)
    deallocate(ttype,tform,tunit)

    ! store growth-curve
    call ftmnhd(15,BINARY_TBL,GROWCURVEXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       status = 0
    else
       ! already presented ? remove it !
       call ftdhdu(15,hdutype,status)
       if( status /= 0 ) goto 666
    end if

    n = 4
    allocate(ttype(n), tform(n), tunit(n))

    tform = '1D'
    tunit = ''
    ttype(1) = FITS_COL_R
    ttype(2) = FITS_COL_GROW
    ttype(3) = FITS_COL_GROWERR
    ttype(4) = FITS_COL_RPROF

    call ftibin(15,0,size(ttype),ttype,tform,tunit,GROWCURVEXTNAME,0,status)
    call ftpcle(15,1,frow,felem,size(raper),raper,status)
    call ftpcld(15,2,frow,felem,size(curve),curve,status)
    call ftpcld(15,3,frow,felem,size(curve),curve_err,status)
    call ftpcld(15,4,frow,felem,size(prof),prof,status)
    deallocate(ttype,tform,tunit)

666 continue

    call fits_close_file(15,status)
    call ftrprt('STDERR',status)

  end subroutine fits_find_save

end module fitsgphot
