/*
 *  Copyright (c) 2005, The Joust Project
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *  - Neither the name of the Joust Project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 *  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 *  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 *
 */

package net.kano.joustsim;

import java.util.List;
import java.util.Collection;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.lang.reflect.Proxy;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

public class JavaTools {
    private JavaTools() { }

    private static final Logger LOGGER = Logger
            .getLogger(JavaTools.class.getName());

    public static void throwExceptions(List<Exception> exceptions,
            String msg) {
        if (exceptions.size() == 1) {
            Exception exception = exceptions.get(0);
            if (exception instanceof RuntimeException) {
                throw (RuntimeException) exception;
            } else {
                throw new IllegalStateException(exception);
            }

        } else if (!exceptions.isEmpty()) {
            for (Exception exception : exceptions) {
                LOGGER.log(Level.WARNING,
                        msg, exception);
            }
            throw new MultipleExceptionsException(exceptions);
        }
    }

    public static <L> L getDelegatingProxy(
            final Collection<? extends L> listeners,
            Class<? extends L> cls) {
        return cast(cls, Proxy.newProxyInstance(cls.getClassLoader(),
                new Class<?>[]{cls},
                new InvocationHandler() {
                    public Object invoke(Object proxy, Method method,
                            Object[] args)
                            throws Throwable {
                        if (!method.getReturnType().equals(Void.TYPE)) {
                            throw new IllegalArgumentException(
                                    "I can't delegate call to " + method
                                            + ": it has non-void return "
                                            + "type");
                        }
                        for (L l : listeners) {
                            method.invoke(l, args);
                        }
                        return null;
                    }
                }));
    }

  @SuppressWarnings({"unchecked"})
  public static <E> E cast(Class<E> cls, Object o) {
    return (E) o;
  }
}
