/*
    Copyright 2015 Vladimir Glazachev

    This file is part of FLINT.

    FLINT is free software: you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.  See <https://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include "profiler.h"
#include "aprcl.h"

void p_is_prime_repeat(fmpz_t * ns, int len)
{
    int i;

    TIMEIT_START
    for (i = 0; i < len; i++)
            if (aprcl_is_prime(ns[i]) == 0)
            {
                flint_printf("Primality test failed\n");
                abort();
            }
    TIMEIT_STOP
}

int main(void)
{
    const int len = 5;
    FLINT_TEST_INIT(state);

    /*
        Using the primes from:
        https://primes.utm.edu/lists/small/
    */

    flint_printf("Primality test profiling for numbers from 10 to 300 digits\n");
    flint_printf("All timings given for 5 numbers\n");

    /* 10 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "5915587277", 10);
        fmpz_set_str(ns[1], "9576890767", 10);
        fmpz_set_str(ns[2], "5463458053", 10);
        fmpz_set_str(ns[3], "3367900313", 10);
        fmpz_set_str(ns[4], "2860486313", 10);

        flint_printf("10 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 20 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "48112959837082048697", 10);
        fmpz_set_str(ns[1], "29497513910652490397", 10);
        fmpz_set_str(ns[2], "71755440315342536873", 10);
        fmpz_set_str(ns[3], "66405897020462343733", 10);
        fmpz_set_str(ns[4], "12764787846358441471", 10);

        flint_printf("20 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 30 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "671998030559713968361666935769", 10);
        fmpz_set_str(ns[1], "590872612825179551336102196593", 10);
        fmpz_set_str(ns[2], "416064700201658306196320137931", 10);
        fmpz_set_str(ns[3], "115756986668303657898962467957", 10);
        fmpz_set_str(ns[4], "280829369862134719390036617067", 10);

        flint_printf("30 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 40 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "2425967623052370772757633156976982469681", 10);
        fmpz_set_str(ns[1], "6075380529345458860144577398704761614649", 10);
        fmpz_set_str(ns[2], "4384165182867240584805930970951575013697", 10);
        fmpz_set_str(ns[3], "6847944682037444681162770672798288913849", 10);
        fmpz_set_str(ns[4], "5570373270183181665098052481109678989411", 10);

        flint_printf("40 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 50 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "22953686867719691230002707821868552601124472329079", 10);
        fmpz_set_str(ns[1], "46484729803540183101830167875623788794533441216779", 10);
        fmpz_set_str(ns[2], "64495327731887693539738558691066839103388567300449", 10);
        fmpz_set_str(ns[3], "15452417011775787851951047309563159388840946309807", 10);
        fmpz_set_str(ns[4], "95647806479275528135733781266203904794419563064407", 10);

        flint_printf("50 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 60 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "622288097498926496141095869268883999563096063592498055290461", 10);
        fmpz_set_str(ns[1], "668486051696691190102895306426999370394054817506916629001851", 10);
        fmpz_set_str(ns[2], "361720912810755408215708460645842859722715865206816237944587", 10);
        fmpz_set_str(ns[3], "511704374946917490638851104912462284144240813125071454126151", 10);
        fmpz_set_str(ns[4], "470287785858076441566723507866751092927015824834881906763507", 10);

        flint_printf("60 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 70 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "4669523849932130508876392554713407521319117239637943224980015676156491", 10);
        fmpz_set_str(ns[1], "2409130781894986571956777721649968801511465915451196376269177305066867", 10);
        fmpz_set_str(ns[2], "7595009151080016652449223792726748985452052945413160073645842090827711", 10);
        fmpz_set_str(ns[3], "5885903965180586669073549360644800583458138238012033647539649735017287", 10);
        fmpz_set_str(ns[4], "9547848065153773335707495885453566120069130270246768806790708393909999", 10);

        flint_printf("70 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 80 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "18532395500947174450709383384936679868383424444311405679463280782405796233163977", 10);
        fmpz_set_str(ns[1], "54875133386847519273109693154204970395475080920935355580245252923343305939004903", 10);
        fmpz_set_str(ns[2], "40979218404449071854385509743772465043384063785613460568705289173181846900181503", 10);
        fmpz_set_str(ns[3], "19469495355310348270990592580191998639221450743640952620236903851789700309402857", 10);
        fmpz_set_str(ns[4], "67120333368520272532940669112228025474970578938046280618394371551488988323794243", 10);

        flint_printf("80 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 90 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "282755483533707287054752184321121345766861480697448703443857012153264407439766013042402571", 10);
        fmpz_set_str(ns[1], "463199005416013829210323411514132845972525641604435693287586851332821637442813833942427923", 10);
        fmpz_set_str(ns[2], "664869143773196608462001772779382650311673568542237852546715913135688434614731717844868261", 10);
        fmpz_set_str(ns[3], "204005728266090048777253207241416669051476369216501266754813821619984472224780876488344279", 10);
        fmpz_set_str(ns[4], "976522637021306403150551933319006137720124048624544172072735055780411834104862667155922841", 10);

        flint_printf("90 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 100 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "2074722246773485207821695222107608587480996474721117292752992589912196684750549658310084416732550077", 10);
        fmpz_set_str(ns[1], "5371393606024775251256550436773565977406724269152942136415762782810562554131599074907426010737503501", 10);
        fmpz_set_str(ns[2], "2908511952812557872434704820397229928450530253990158990550731991011846571635621025786879881561814989", 10);
        fmpz_set_str(ns[3], "7212610147295474909544523785043492409969382148186765460082500085393519556525921455588705423020751421", 10);
        fmpz_set_str(ns[4], "6513516734600035718300327211250928237178281758494417357560086828416863929270451437126021949850746381", 10);

        flint_printf("100 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 110 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "10513733234846849736873637829838635104309714688896631127438692162131857778044158273164093838937083421380041997", 10);
        fmpz_set_str(ns[1], "24684249032065892333066123534168930441269525239006410135714283699648991959894332868446109170827166448301044689", 10);
        fmpz_set_str(ns[2], "32998886283809577512914881459957314326750856532546837122557861905096711274255431870995137546575954361422085081", 10);
        fmpz_set_str(ns[3], "36261430139487433507414165833468680972181038593593271409697364115931523786727274410257181186996611100786935727", 10);
        fmpz_set_str(ns[4], "76921421106760125285550929240903354966370431827792714920086011488103952094969175731459908117375995349245839343", 10);

        flint_printf("110 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 120 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "171989218832470870857923701742780850419373475175043795375607676047792517311566339217543754147263751973899368604441353849", 10);
        fmpz_set_str(ns[1], "371132472088173209741153184981742771278849120163424101995797949338636074962048027958518451774716413729510755717494155299", 10);
        fmpz_set_str(ns[2], "420145406901811857791227072284165226561693483222287527567496017033892563342686752247587935117119306171161848593337649107", 10);
        fmpz_set_str(ns[3], "506283312611011343355256478253272463245968105632679003983305635125224133331073348553775052064302641255435067238306718511", 10);
        fmpz_set_str(ns[4], "897503957504227472139484199430066010338139343163145419280183314291067450988520718807102741476596034735471026312154231263", 10);

        flint_printf("120 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 130 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "1182911141497541140586615359230243592834259935951457923890100901143724702295750543180757886003762250822014770432406785975220712551", 10);
        fmpz_set_str(ns[1], "3007481519774841700447169103699480479257075158004429172505493169624211738165626204221653115518171895124254955050192476647393240549", 10);
        fmpz_set_str(ns[2], "4578487274208482739785211870551459143579948064875277467213620367693698827278583879146069992097475604136147868628836746016239588907", 10);
        fmpz_set_str(ns[3], "6539010935836249262943170849578201335017375258186767902505297465587122376751466876714291620849745334794779375900778144335240097113", 10);
        fmpz_set_str(ns[4], "8714772976381697859557965355791745997987728300226812732519702250463329265826033268926171610392211750044957719896320776794555146581", 10);

        flint_printf("130 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 140 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "11972074619991162245806805523618114505186632677199227008626207635331666111231564033315528198753090956109606784942142173066807025406537144269", 10);
        fmpz_set_str(ns[1], "21384673207444598953944295269132432413002706756010583952155931098725644623031415143234836631619334100621702323118864487149301235435577193841", 10);
        fmpz_set_str(ns[2], "40715419051417482519286539367794251058952189812236314107150220232148457082328798652774511688107134820082466929596616575469866935012183992821", 10);
        fmpz_set_str(ns[3], "49105133753678962419391374496727457097027115514864975959094375339860082156189031654878635228598019107441653554142227931036658923353143622433", 10);
        fmpz_set_str(ns[4], "66584720460899785255133191060974440278633915171229599356820775802394462756364476093414927157574862478083047015999630105316838183873438847653", 10);

        flint_printf("140 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 150 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "204616454475328391399619135615615385636808455963116802820729927402260635621645177248364272093977747839601125961863785073671961509749189348777945177811", 10);
        fmpz_set_str(ns[1], "291550339212810520584591717024643003222896524222464547399033563270236499245759688372706177143834849792024373534800701771810502419549113758796692666163", 10);
        fmpz_set_str(ns[2], "375558729886621567205449255384643401545510736101936314566046534513364279097663141755273454199533071264570131949667792104742749053406471675458926748329", 10);
        fmpz_set_str(ns[3], "533791764536500962982816454877600313815808544134584704665367971790938714376754987723404131641943766815146845004667377003395107827504566198008424339207", 10);
        fmpz_set_str(ns[4], "656692050181897513638241554199181923922955921760928836766304161790553989228223793461834703506872747071705167995972707253940099469869516422893633357693", 10);

        flint_printf("150 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 160 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "1081298104698286063813737967304568031406522676857739555339880517562925221530558524296599584286163751908713364829390795648074146197550782524900963175263757603219", 10);
        fmpz_set_str(ns[1], "3453221202185876906638629632175359499099670410652382616441386629981117932849225524689308950511055601388422752776562037360817653410867799137900072936906881621341", 10);
        fmpz_set_str(ns[2], "5527520013513581326121882167187180752073463254277609911934750350965822023838543172229709244319844670433804038920304644097364765738975596420673642099039975011363", 10);
        fmpz_set_str(ns[3], "8726641590503419650325196619937603416439489172580927797921688450581350540574217626151578525827423033276197489219680752921410238881917731681447358498204217268973", 10);
        fmpz_set_str(ns[4], "9379635662817322614032764342613018955177071738482024436203544111095204370743225356112956525567890031429756904091450073222354327129541982170431394374683183905361", 10);

        flint_printf("160 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 170 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "11067896864151107888327210475457768522190748228130894643297732650458556828346322177345544084702872726772673298479414374934206448595620856513559764913452487779500078246341", 10);
        fmpz_set_str(ns[1], "21160656367574879904377656869692416560544398642776271862237616754494681129688319299114321001429667792412314306472092655735101612562676478693400212218038544113953364215017", 10);
        fmpz_set_str(ns[2], "32559552391090806972524986110685454429554861128146958282307580573550839559287659656080789996587490910999032609385668148399053113719642305511467653200258413635664259044077", 10);
        fmpz_set_str(ns[3], "37745683039627526420239498495450014901855006768892775749597403587571970429951577000073551065765218057239528311184736067680411048526187122288661901211789066482483802580969", 10);
        fmpz_set_str(ns[4], "61666372912245863887684510042656975220774126438327774649153997946400603024380560713336470674573893680427095868805106128483007129797880040347838399734458883689665738771401", 10);

        flint_printf("170 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 180 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "142421590971309087406926930527203509136940618029846108794169863876683171607366571570400885831140555590135576733907110189937788062898855050894112651266072474307203583352058594056699", 10);
        fmpz_set_str(ns[1], "278344420100310216738042441752470108332323908618219525487053710207384771998286597135290418719526702385178365896340210957041552257518693488098683242070746473230980005141419410511409", 10);
        fmpz_set_str(ns[2], "539813721464880304831529082014022773775683060863538731923720954030918288330963319766918459920070168122735288577580167859459419843291482297756648680136952821437700382745720613744889", 10);
        fmpz_set_str(ns[3], "702859775508284030177919822558533006622440218103436237875100165464040768114702545328008730929739757973342358123790248243520711425000515624162703794316531258568583660237867578953203", 10);
        fmpz_set_str(ns[4], "988024003287850859144419046035039824149244726706784277084869779315425892879684597047236357710409517227957670271186515606462572615012822834344038901575679797975183588867579172966813", 10);

        flint_printf("180 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 190 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "2555480246614015637242066218495468367958491958791520916609927980929585688386540022963356429758954681777165591120866518535295021508514987925230304828573354059077193219265153006831360318167711", 10);
        fmpz_set_str(ns[1], "3643214290461749241398376492190485585384494258388704356737409156502890948489887688353135368121060124798243703432841307147301729071453564875847359568200406818058535340069409695144897237869317", 10);
        fmpz_set_str(ns[2], "4837500633209442055641398894165176677263201858612849667774338162796452956137875114054119135158253792745804801331435807680095292929070880966039245167637417105254552418199073193006899620258743", 10);
        fmpz_set_str(ns[3], "5077461915617371651811593607389848534613910665160223569902404065418076795944656510340577164207672426574638345121648036693344561339868694500127046808526397857391514634557427341972391976756821", 10);
        fmpz_set_str(ns[4], "6455677032078134615122923256047783642469165615329330252550539894033032422509072781300599341862693333305369588191523252516876513543095490001221889525858323800754356138220797092839047522855519", 10);

        flint_printf("190 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 200 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "15474811206486587193258690501682404626361341756658894201908294153626080782693777003022566996735796983239343580281979005677758015801189957392350213806122307985157041153484138150252828152419133170303749", 10);
        fmpz_set_str(ns[1], "28591045597720075832628274729885724490653298360003309382769144463123258670807750560985604954275365591715208615509779345682419533206637382048824349415329839450792353652240682445321955199147316594996133", 10);
        fmpz_set_str(ns[2], "41184172451867371867686906412307989908388177848827102865167949679167771021417488428983978626721272105583120243720400358313998904049755363682307706550788498535402989510396285940007396534556364659633739", 10);
        fmpz_set_str(ns[3], "58021664585639791181184025950440248398226136069516938232493687505822471836536824298822733710342250697739996825938232641940670857624514103125986134050997697160127301547995788468137887651823707102007839", 10);
        fmpz_set_str(ns[4], "71611195866368241734230315014260885890178941731009368469658803702463720956633120935294831101757574996161931982864195542669330457046568876289241536680683601749507786059442920003278263334056542642264651", 10);

        flint_printf("200 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 210 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "230975859993204150666423538988557839555560243929065415434980904258310530753006723857139742334640122533598517597674807096648905501653461687601339782814316124971547968912893214002992086353183070342498989426570593", 10);
        fmpz_set_str(ns[1], "110975366417606590833967199024636437460214709654546954443361638176229883140910302915147996598210506836614251268964446338737195911448213015377273363573074752120243818732263682864461580384738021460717565715722503", 10);
        fmpz_set_str(ns[2], "449417999055441493994709297093108513015373787049558499205492347871729927573118262811508386655998299074566974373711472560655026288668094291699357843464363003144674940345912431129144354948751003607115263071543163", 10);
        fmpz_set_str(ns[3], "643808006803554439230129854961492699151386107534013432918073439524138264842370630061369715394739134090922937332590384720397133335969549256322620979036686633213903952966175107096769180017646161851573147596390153", 10);
        fmpz_set_str(ns[4], "964274047248418797145090983157197980855078966882276492572788532954904112655338439361306213898569516593744267391754033306465125919199692703323878557833023573312685002670662846477592597659826113460619815244721311", 10);

        flint_printf("210 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 220 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "1686497341353711305611314533713892867476872069660265164960400834285140167272334456461153902160145656434656190916734048745690082065807224646832960020851742291098965544262163963976377940891936112842489791084937115229985523", 10);
        fmpz_set_str(ns[1], "3082620745243449401694587258115336002381779641367375950273829476342471735062725180901036226278130762025829992363923955707979142964213859746930138447696031315806586014398729784266819509742760398915336761425413832264717163", 10);
        fmpz_set_str(ns[2], "4024609901503260810269407468691203815597240980490831121441008103456384399515806457060473646208324881470459201945576188438333273668474345882626708030947627287590124455488557086128125418355846343248375161824453194911368553", 10);
        fmpz_set_str(ns[3], "5521712099665906221540423207019333379125265462121169655563495403888449493493629943498064604536961775110765377745550377067893607246020694972959780839151452457728855382113555867743022746090187341871655890805971735385789993", 10);
        fmpz_set_str(ns[4], "5569632747079549601598178420932244414970531646975337735765307429600364098163302494170010799604994405157512160053825953960398479355068806977610414154008753377194113913520943719945884635263031964263024822919752634970219699", 10);

        flint_printf("220 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 230 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "21300201301629210452464119866388466834939209799796743744566215971019835934873713391025654608111703142839573952969146866196732314492859634538441409805139418615474981665588472532863079300348061212642715504474934450531890795747236331", 10);
        fmpz_set_str(ns[1], "12245442024374614689109712434724272673056459509314527349744503469345912205668622170940248748257546033159240098557771252551702406748969563113679865954698092482472205671770925537241038679473374235787361928891214113717891603697678797", 10);
        fmpz_set_str(ns[2], "38762217770011076543812558980103749234859497948714246622214162442205108754782773687842018087751465069758232680421784403844939497206445510513337080943785005155511815286799602210383243566108920435480682463875028936997633500773799647", 10);
        fmpz_set_str(ns[3], "56125680981752282333498088313568935051383833838594899821664631784577337171193624243181360054669678410455329112434552942717084003541384594864129940145043086760031292483340068923506115878221189886491132772739661669044958531131327771", 10);
        fmpz_set_str(ns[4], "63556399829835331952308746203320003046976892440596186268797010059920148204359911235920191263570262537797901436399163646602402069151710265676725083444680749168670016260057105903268204928263311884485983112346974646808220605856122413", 10);

        flint_printf("230 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 240 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "120815570979701484704906977000760567182871429114712069861589084706550626575967516787438008593490722779337547394120718248995900363209947025063336882559539208430319216688889117222633155838468458047056355241515415159736436403445579777425189969", 10);
        fmpz_set_str(ns[1], "255112508897622985310494621898578485506502699374200456279537176322071752799387264311493082040326094068304025184818529912661589221508006459487667465358421280596050439152052898117848487594823639685476220238281599236511128329403379244547324431", 10);
        fmpz_set_str(ns[2], "429658446338661686643493725510284110493599954226035820905785008443520381887917251101430906695273248229134343944968312754799687049087615143654956740908022486161773863760198918360760883640680471523406895493680359160538947211150254456062366151", 10);
        fmpz_set_str(ns[3], "658385546911733550164516088405238961461880256029834598831972039469421755117818013653494814438931957316403111689187691446941406788869098983929874080332195117465344344350008880118042764943201875870917468833709791733282363323948005998269792207", 10);
        fmpz_set_str(ns[4], "963776490226810914179127416979112736599358462343660979133704637052985987395814985567964199253678349130533981848158045863245610061240372125223976414165504075241141636320276071785214612776961231843830877454050632964032066734998244033965101749", 10);

        flint_printf("240 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 250 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "1389116249243716847348088927848659359447606893455263094651000788996986658678996011997982077310912124848525958777046944012259203097866174509363711195882419322015576431662790538354378481778662165275630720476067089175069283834501499113947948000501538579", 10);
        fmpz_set_str(ns[1], "3416125484651880824649192099235033527914617803403701054777894769025318772710175775793367341953921714553521186823810932269184143033013016793222402163346724187114477766138018341566351560693965548954433712778344922065660960958894751416275568079834471541", 10);
        fmpz_set_str(ns[2], "5236828875479032067717626462856300906381035007260718615838669162211131824655260246937578375433262328206499459275581209388342892208784057258717047490425447217362253059951830269996579032687344642379285373251685453043862105987293264183501516155981651569", 10);
        fmpz_set_str(ns[3], "5781538327977828897150909166778407659250458379645823062042492461576758526757490910073628008613977550546382774775570888130029763571528699574717583228939535960234464230882573615930384979100379102915657483866755371559811718767760594919456971354184113721", 10);
        fmpz_set_str(ns[4], "6325427919960049066585247837578372385418559154923477553398129089734082978096069693032859784967901396775824948679013568274245239986849282715816927424255093730637896848500237375779410539868274303393510928400955586603945601202203906813552017713600173613", 10);

        flint_printf("250 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 260 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "16107407153096872818493782945957787561325375534122812732915532201840816254400467958193423121840372946330339589450123508622134774121104116098949168616561321246035895208358144099325905467103390396346965217633409955370491728893923457194644438929148135544797769069", 10);
        fmpz_set_str(ns[1], "25290181317443223202462289639535568588026980307751030325846138922434211534444969788237980189518561067454419530259429589051181436567403701018238964718203056168955426581729913704293743890489851209326371108494646496297959925452431230546049667449322574935832180327", 10);
        fmpz_set_str(ns[2], "49642196519287563890219771644794994818956271978536893649678493707106697740616207443433039867021200759433868948129547546653564382798817640018032929084872815787149827529492755939735263141371178259217768571082104248486704273772193856448675375667327678108529811271", 10);
        fmpz_set_str(ns[3], "50985706168761488366316382338014644105350894561341788683522249832194333902444754897729974158787863583439645933374661244079621367847162487366917832904994676462834272619451739021241059378594481762281376106663491397418420352316354537398792605354478235631207230857", 10);
        fmpz_set_str(ns[4], "67456772536997811174562843827334166492604722726798354062809571765666143205478154891520988418820829227136251602140592962830914859588069638333300746582776692720637910854613289610057151587251089921722318897093966945777208101090776716402383331743368026418397556891", 10);

        flint_printf("260 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 270 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "170902679027143614381480059391075460977748667084652490023526958676379938833341709136279594902205355969210787455441130248035902265317439239727527609176424101012027295635928382169934598883924161721713564178522447830194403641544177926621519514748243624424803053704080264159", 10);
        fmpz_set_str(ns[1], "341542941372464534497852047268105389138631890905646486032000153894901449540580129450122392139802619604292752498326041347480618250352488051163893646047131867622060323720251466252698326430731130611964327062361372397940193111039277572779594686777531915056404892709536037639", 10);
        fmpz_set_str(ns[2], "471475783779117473437696117802417626487839062434599064863035821968629035024980502377134378815374602544529087517402940868050418215262184298226449528574182158863791371813026718349131181672601351927241854659801579851938035856589600671154437708549060762345454578611174965613", 10);
        fmpz_set_str(ns[3], "533908591185102096721484909806413036775185183352706309151302910709983373762043847868389077713822984385951546710427861029531444058770240205707393053275752747465332797251246740378632048744214547340674874776350282974442698782288381468714759061146399901895491734760806078183", 10);
        fmpz_set_str(ns[4], "630401459981890925871943502656990828862174194616722524583664885334493959070286140976851072711744928475299148378810029026986307409791403597358746752833626753147802185593071445295784983543019670778723774980356622160978299480879244289586779357458029514963543350008665423257", 10);

        flint_printf("270 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 280 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "2472773490088428133980402514441160210616362471988372788573252776560313375548879579643617872035716953544469744723310736110488902356856738414994328039965984433249491556806522755609731304100884485833876145266471071156842673554540732151138408860390316340855518308224087245071159274479", 10);
        fmpz_set_str(ns[1], "4502627151660440701021925419329909839338324685276222991558243422906343244255978228150075701850481154561057205207514930885060886603445774420800514839637354644396387390585531622698158148515281319634118340188729326221225299940392408901441649535445332397302718287942088535978634236221", 10);
        fmpz_set_str(ns[2], "5433381957526079619502954301327211367491752117714541408014490376067456113285722791097965934635155872134552184960882819005997047037814265447050608697209373083125903651449532919407946144950853232347970900124595900080331923968470611977246337077886181977132283382211753065664038103921", 10);
        fmpz_set_str(ns[3], "6761310901839275572586265703525965167391700024715455969978251868036124196491638874957886928031547478720430809462609191151492313143792903392719106050713843902317385520764341022452298421899503012580175228138841759425959660620124299740103602713372417764600890682511057074682657737071", 10);
        fmpz_set_str(ns[4], "8036474984942099540833063531872969308026120828502186792023446855615691106313478083033164185643655131177550064421680055607368972403502030505525586634272348294932475045273750810461772447879143316699989478715170989152555192208636375418186997456472778736820533608167526171909181155851", 10);

        flint_printf("280 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 290 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "16471581891701794764704009719057349996270239948993452268812975037240586099924712715366967486587417803753916334331355573776945238871512026832810626226164346328807407669366029926221415383560814338828449642265377822759768011406757061063524768140567867350208554439342320410551341675119078050953", 10);
        fmpz_set_str(ns[1], "28911710017320205966167820725313234361535259163045867986277478145081076845846493521348693253530011243988160148063424837895971948244167867236923919506962312185829914482993478947657472351461336729641485069323635424692930278888923450060546465883490944265147851036817433970984747733020522259537", 10);
        fmpz_set_str(ns[2], "36997438997453671132166865292569847244619444010395996997269402633033171733524207782461697123413414988859119764706460324782855919613943534861045216286685374559389860714010358938897470990143322775796082401132386180536650715835089633575578638914905606853502795766373129779823319637155282395679", 10);
        fmpz_set_str(ns[3], "47636001527157843615510166362212813825553995166349359052560828063795751982743949356281378571281356684995526346349689115319412565735006540850254022130876377867983398374239198108049044484863543699346022227692817853215225730056764957039852693911896900286014520659611654434776741074401112976951", 10);
        fmpz_set_str(ns[4], "66411949393245949268811711602826765576402057646975003006251042260813215340087318062380031915073315092183806206493533345953281647263552710655695269967337089850144857674638489475995919778552032603791816048265084649175429768094838031170157033168866305251844356097795408000548418783227194651709", 10);

        flint_printf("290 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    /* 300 digits primes */
    {
        int i;
        fmpz_t * ns;

        ns = (fmpz_t*) flint_malloc(sizeof(fmpz_t) * len);

        for (i = 0; i < 5; i++)
            fmpz_init(ns[i]);

        fmpz_set_str(ns[0], "203956878356401977405765866929034577280193993314348263094772646453283062722701277632936616063144088173312372882677123879538709400158306567338328279154499698366071906766440037074217117805690872792848149112022286332144876183376326512083574821647933992961249917319836219304274280243803104015000563790123", 10);
        fmpz_set_str(ns[1], "250556952327646214427246777488032351712139094643988394726193347352092526616305469220133287929222242315761834129196430398011844978805263868522770723615504744438638381670321613949280530254014602887707960375752016807510602846590492724216092721283154099469988532068424757856392563537802339735359978831013", 10);
        fmpz_set_str(ns[2], "290245329165570025116016487217740287508837913295571609463914348778319654489118435855243301969001872061575755804802874062021927719647357060447135321577028929269578574760547268310055056867386875959045119093967972205124270441648450825188877095173754196346551952542599226295413057787340278528252358809329", 10);
        fmpz_set_str(ns[3], "319705304701141539155720137200974664666792526059405792539680974929469783512821793995613718943171723765238853752439032835985158829038528214925658918372196742089464683960239919950882355844766055365179937610326127675178857306260955550407044463370239890187189750909036833976197804646589380690779463976173", 10);
        fmpz_set_str(ns[4], "531872289054204184185084734375133399408303613982130856645299464930952178606045848877129147820387996428175564228204785846141207532462936339834139412401975338705794646595487324365194792822189473092273993580587964571659678084484152603881094176995594813302284232006001752128168901293560051833646881436219", 10);

        flint_printf("300 digit primes\n");
        p_is_prime_repeat(ns, len);

        for (i = 0; i < 5; i++)
            fmpz_clear(ns[i]);
        flint_free(ns);
    }

    return 0;
}

