/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file pwd_gnome.c
 * \brief GNOME Keyring plugin: Store and find of passwords in gnome keyring
 */

#include <ffgtk.h>
#include <gnome-keyring-1/gnome-keyring.h>
#include <gnome-keyring-1/gnome-keyring-result.h>

/** ffgtk password schemea, containing app name, profile name */
static GnomeKeyringPasswordSchema sSchema = {
    GNOME_KEYRING_ITEM_GENERIC_SECRET, {
        { "app", GNOME_KEYRING_ATTRIBUTE_TYPE_STRING },
        { "profile", GNOME_KEYRING_ATTRIBUTE_TYPE_STRING },
        { NULL, 0 },
    }
};

static gchar *pnSecretPassword = NULL;

static const gchar *gnomeKeyringGetPassword( struct sProfile *psProfile ) {
	GnomeKeyringResult nRet;

	if ( pnSecretPassword != NULL ) {
		return pnSecretPassword;
	}

	if ( psProfile == NULL || psProfile -> pnName == NULL || gnome_keyring_is_available() == FALSE ) {
		return NULL;
	}

	/* Find password */
	nRet = gnome_keyring_find_password_sync(
		&sSchema,
		&pnSecretPassword,
		"app", "ffgtk",
		"profile", psProfile -> pnName,
		NULL
	);

	/* Check return value, if no error occurred: return password */
	if ( nRet == GNOME_KEYRING_RESULT_OK ) {
		return pnSecretPassword;
	}

	Debug( KERN_WARNING, "Couldn't find password: %s\n", gnome_keyring_result_to_message( nRet ) );
	pnSecretPassword = NULL;

	return NULL;
}

/**
 * \brief Set password to gnome keyring
 * \param psProfile profile structure
 * \param pnPassword profile password
 */
static void gnomeKeyringSetPassword( struct sProfile *psProfile, const gchar *pnPassword ) {
	GnomeKeyringResult nRet;
	gchar *pnName = NULL;

	if ( psProfile == NULL || psProfile -> pnName == NULL || gnome_keyring_is_available() == FALSE ) {
		return;
	}

	pnName = g_strdup_printf( "ffgtk password for profile '%s'", psProfile -> pnName );

	/* store password */
	nRet = gnome_keyring_store_password_sync(
		&sSchema,
		GNOME_KEYRING_DEFAULT,
		pnName,
		pnPassword,
		"app", "ffgtk",
		"profile", psProfile -> pnName,
		NULL
	);

	/* Check return value, if error occurred: show reason */
	if ( nRet != GNOME_KEYRING_RESULT_OK ) {
		Debug( KERN_WARNING, "Couldn't store password: %s\n", gnome_keyring_result_to_message( nRet ) );
	} else {
		if ( pnSecretPassword != NULL ) {
			gnome_keyring_free_password( pnSecretPassword );
		}
		pnSecretPassword = NULL;
	}

	/* free name */
	g_free( pnName );
}


/**
 * \brief Remove password from gnome keyring
 * \param psProfile profile structure
 * \param pnPassword profile password
 */
static void gnomeKeyringRemovePassword( struct sProfile *psProfile ) {
	GnomeKeyringResult nRet;

	if ( psProfile == NULL || psProfile -> pnName == NULL || gnome_keyring_is_available() == FALSE ) {
		return;
	}

	nRet = gnome_keyring_delete_password_sync(
		&sSchema,
		"app", "ffgtk",
		"profile", psProfile -> pnName,
		NULL
	);

	/* Check return value, if error occurred: show reason */
	if ( nRet != GNOME_KEYRING_RESULT_OK ) {
		Debug( KERN_WARNING, "Couldn't remove password: %s\n", gnome_keyring_result_to_message( nRet ) );
	} else {
		if ( pnSecretPassword != NULL ) {
			gnome_keyring_free_password( pnSecretPassword );
		}
		pnSecretPassword = NULL;
	}
}

/** password definition */
struct sPassword sKeyring = {
	gnomeKeyringGetPassword,
	gnomeKeyringSetPassword,
	gnomeKeyringRemovePassword
};

MODULE_INIT( PLUGIN_TYPE_PASSWORD, _( "GNOME Keyring" ), &sKeyring, NULL, NULL );
