/*******************************************************************
 * libfaxophone                                                    *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file fax.c
 * \brief Fax handling functions
 */

#include "faxophone.h"
#include "fax.h"
#include "isdn-convert.h"

static int8_t *_linear16_2_law = &linear16_2_law[ 32768 ];
static uint16_t *_law_2_linear16 = &law_2_linear16[ 0 ];

static int nLogLevel = 0;

static void (*logging)( int nLevel, const char *pnText ) = NULL;

/**
 * \brief Dump spandsp messages
 * \param nLevel spandsp loglevel
 * \param pnText message text
 */
static void spandsp_msg_log( int nLevel, const char *pnText ) {
	FopDebug( FOP_DEBUG, "%s", pnText );
}

/**
 * \brief Realtime frame handler which keeps tracks of data transfer
 * \param psState t30 state pointer
 * \param pUserData pointer to connection index
 * \param nDirection transmission direction
 * \param pnMsg spandsp message
 * \param nLen len of frame
 */
static void realTimeFrameHandler( t30_state_t *psState, void *pUserData, int nDirection, const uint8_t *pnMsg, int nLen ) {
	struct sCapiConnection *psConnection = pUserData;
	struct sFaxStatus *psStatus = psConnection -> pPrivate;
	struct sSession *psSession = faxophoneGetSession();
	t30_stats_t sStats;

	FopDebug( FOP_DEBUG, "realTimeFrameHandler() called (%d/%d/%d)\n", nDirection, nLen, pnMsg[ 2 ] );
	if ( pnMsg[ 2 ] == 6 ) {
		t30_get_transfer_statistics( psState, &sStats );

		if ( psStatus -> nSending ) {
			psStatus -> nBytesTotal = sStats.image_size;
			psStatus -> nBytesSent += nLen;
		} else {
			psStatus -> nBytesReceived += nLen;
			psStatus -> nBytesTotal += nLen;
		}

		psSession -> psHandlers -> Status( psConnection, 1 );
	}
}

/**
 * \brief Phase B handler
 * \param psState t30 state pointer
 * \param pUserData pointer to connection
 * \param nResult result
 * \return error code
 */
static int phaseHandlerB( t30_state_t *psState, void *pUserData, int nResult ) {
	struct sCapiConnection *psConnection = pUserData;
	struct sFaxStatus *psStatus = psConnection -> pPrivate;
	struct sSession *psSession = faxophoneGetSession();
	t30_stats_t sStats;
	t30_state_t *psT30;

	t30_get_transfer_statistics( psState, &sStats );
	psT30 = fax_get_t30_state( psStatus -> psFaxState );

	FopDebug( FOP_DEBUG, "Phase B handler (0x%X) %s\n", nResult, t30_frametype( nResult ) );
	FopDebug( FOP_DEBUG, " - bit rate %d\n", sStats.bit_rate );
	FopDebug( FOP_DEBUG, " - nEcm %s\n", ( sStats.error_correcting_mode )  ?  "on"  :  "off" );

	if ( psStatus -> nSending ) {
		snprintf( psStatus -> anIdent, sizeof( psStatus -> anIdent ), "%s", t30_get_rx_ident( psT30 ) );
	} else {
		snprintf( psStatus -> anIdent, sizeof( psStatus -> anIdent ), "%s", t30_get_tx_ident( psT30 ) );
	}
	FopDebug( FOP_DEBUG, "Remote side: '%s'\n", psStatus -> anIdent );
	if ( t30_get_rx_sender_ident( psT30 ) ) {
		FopDebug( FOP_DEBUG, "Remote side sender: '%s'\n", t30_get_rx_sender_ident( psT30 ) );
	}
	if ( t30_get_rx_country( psT30 ) ) {
		FopDebug( FOP_DEBUG, "Remote side country: '%s'\n", t30_get_rx_country( psT30 ) );
	}
	if ( t30_get_rx_vendor( psT30 ) ) {
		FopDebug( FOP_DEBUG, "Remote side vendor: '%s'\n", t30_get_rx_vendor( psT30 ) );
	}
	if ( t30_get_rx_model( psT30 ) ) {
		FopDebug( FOP_DEBUG, "Remote side model: '%s'\n", t30_get_rx_model( psT30 ) );
	}

	psStatus -> ePhase = PHASE_B;
	psStatus -> nBytesSent = 0;
	psStatus -> nBytesTotal = 0;
	psStatus -> nBytesReceived = 0;
	psStatus -> nEcm = sStats.error_correcting_mode;
	psStatus -> nBadRows = sStats.bad_rows;
	psStatus -> nEncoding = sStats.encoding;
	psStatus -> nBitrate = sStats.bit_rate;
	psStatus -> nPageTotal = sStats.pages_in_file;

	psStatus -> nPageCurrent = psStatus -> nSending ? sStats.pages_tx + 1 : sStats.pages_rx + 1;
	if ( psStatus -> nSending ) {
		psStatus -> nBytesSent = 0;
	}

	psSession -> psHandlers -> Status( psConnection, 0 );

	return 0;
}

/**
 * \brief Phase D handler
 * \param psState t30 state pointer
 * \param pUserData pointer to connection
 * \param nResult result
 * \return error code
 */
static int phaseHandlerD( t30_state_t *psState, void *pUserData, int nResult ) {
	struct sCapiConnection *psConnection = pUserData;
	struct sFaxStatus *psStatus = psConnection -> pPrivate;
	struct sSession *psSession = faxophoneGetSession();
	t30_stats_t sStats;

	t30_get_transfer_statistics( psState, &sStats );

	FopDebug( FOP_DEBUG, "Phase D handler (0x%X) %s\n", nResult, t30_frametype( nResult ) );
	FopDebug( FOP_DEBUG, " - pages transferred %d\n", psStatus -> nSending ? sStats.pages_tx : sStats.pages_rx );
	FopDebug( FOP_DEBUG, " - image size %d x %d\n", sStats.width, sStats.length );
	FopDebug( FOP_DEBUG, " - bad rows %d\n", sStats.bad_rows );
	FopDebug( FOP_DEBUG, " - longest bad row run %d\n", sStats.longest_bad_row_run );
	FopDebug( FOP_DEBUG, " - image size %d\n", sStats.image_size );

	psStatus -> ePhase = PHASE_D;
	psStatus -> nEcm = sStats.error_correcting_mode;
	psStatus -> nBadRows = sStats.bad_rows;
	psStatus -> nEncoding = sStats.encoding;
	psStatus -> nBitrate = sStats.bit_rate;

	if ( psStatus -> nSending ) {
		psStatus -> nPageCurrent = ( sStats.pages_in_file >= sStats.pages_tx + 1 ? sStats.pages_tx + 1 : sStats.pages_tx );
	} else {
		psStatus -> nPageCurrent = sStats.pages_rx;
	}

	psStatus -> nPageTotal = sStats.pages_in_file;
	psStatus -> nBytesReceived = 0;
	psStatus -> nBytesSent = 0;

	psSession -> psHandlers -> Status( psConnection, 0 );

	return 0;
}

/**
 * \brief Phase E handler
 * \param psState T30 state
 * \param pUserData pointer to current capi connection
 * \param nResult result code
 */
static void phaseHandlerE(t30_state_t *psState, void *pUserData, int nResult ) {
	struct sCapiConnection *psConnection = pUserData;
	struct sFaxStatus *psStatus = psConnection -> pPrivate;
	struct sSession *psSession = faxophoneGetSession();
	int nTransferred = 0;
	t30_stats_t sStats;
	t30_state_t *psT30;

	if ( psStatus -> ePhase == FAX_OK || psState == NULL ) {
		return;
	}

	t30_get_transfer_statistics(psState, &sStats);

	FopDebug( FOP_DEBUG, "Phase E handler (0x%X) %s\n", nResult, t30_completion_code_to_str( nResult ) );

	nTransferred = psStatus -> nSending ? sStats.pages_tx : sStats.pages_rx;
	FopDebug( FOP_DEBUG, " - pages transferred %d\n", nTransferred );
	FopDebug( FOP_DEBUG, " - image resolution %d x %d\n", sStats.x_resolution, sStats.y_resolution );
	FopDebug( FOP_DEBUG, " - compression type %d\n", sStats.encoding );
	FopDebug( FOP_DEBUG, " - coding method %s\n", t4_encoding_to_str( sStats.encoding ) );

	psStatus -> ePhase = PHASE_E;
	psStatus -> nEcm = sStats.error_correcting_mode;
	psStatus -> nBadRows = sStats.bad_rows;
	psStatus -> nEncoding = sStats.encoding;
	psStatus -> nBitrate = sStats.bit_rate;

	psStatus -> nPageCurrent = ( psStatus -> nSending ? sStats.pages_tx : sStats.pages_rx );
	psStatus -> nPageTotal = sStats.pages_in_file;
	psStatus -> nErrorCode = nResult;

	psT30 = fax_get_t30_state( psStatus -> psFaxState );
	if ( psStatus -> nSending ) {
		snprintf( psStatus -> anIdent, sizeof( psStatus -> anIdent ), "%s", t30_get_rx_ident( psT30 ) );
	} else {
		snprintf( psStatus -> anIdent, sizeof( psStatus -> anIdent ), "%s", t30_get_tx_ident( psT30 ) );
	}

	FopDebug( FOP_DEBUG, "Remote station id: %s\n", psStatus -> anIdent );

	FopDebug( FOP_DEBUG, "Fax transfer result (%d) %s\n", nResult, t30_completion_code_to_str( nResult ) );

	psSession -> psHandlers -> Status( psConnection, 0 );
}

/**
 * \brief Initialize spandsp
 * \param pnTiffFile tiff file
 * \param nSending sending flag
 * \param nModem supported modem
 * \param nEcm error correction mode flag
 * \param pnLsi lsi
 * \param pnLocalHeaderInfo local header
 * \param psConnection capi connection poiner
 * \return error code
 */
int spandsp_init( const char *pnTiffFile, int nSending, int nModem, int nEcm, const char *pnLsi, const char *pnLocalHeaderInfo, struct sCapiConnection *psConnection ) {
	t30_state_t *psT30 = NULL;
	logging_state_t *psLogging;
	int nSupportedCompressions = 0;
	int nSupportedResolutions = 0;
	int nSupportedImageSizes = 0;
	int nSupportedModems = 0;
	struct sFaxStatus *psStatus = psConnection -> pPrivate;

	psStatus -> psFaxState = fax_init( psStatus -> psFaxState, nSending );

	fax_set_transmit_on_idle( psStatus -> psFaxState, TRUE );
	fax_set_tep_mode( psStatus -> psFaxState, FALSE );

	psT30 = fax_get_t30_state( psStatus -> psFaxState );

	/* Supported compressions */
	nSupportedCompressions = 0;
	nSupportedCompressions |= T30_SUPPORT_NO_COMPRESSION;
	nSupportedCompressions |= T30_SUPPORT_T4_1D_COMPRESSION;
	nSupportedCompressions |= T30_SUPPORT_T4_2D_COMPRESSION;
	if ( nEcm ) {
		nSupportedCompressions |= T30_SUPPORT_T6_COMPRESSION;
	}

	/* Supported resolutions */
	nSupportedResolutions = 0;
	nSupportedResolutions |= T30_SUPPORT_STANDARD_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_FINE_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_SUPERFINE_RESOLUTION;
	/*nSupportedResolutions |= T30_SUPPORT_R4_RESOLUTION;*/
	nSupportedResolutions |= T30_SUPPORT_R8_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_R16_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_300_300_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_400_400_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_600_600_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_1200_1200_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_300_600_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_400_800_RESOLUTION;
	nSupportedResolutions |= T30_SUPPORT_600_1200_RESOLUTION;

	/* Supported image sizes */
	nSupportedImageSizes = 0;
	nSupportedImageSizes |= T30_SUPPORT_215MM_WIDTH;
	nSupportedImageSizes |= T30_SUPPORT_255MM_WIDTH;
	nSupportedImageSizes |= T30_SUPPORT_303MM_WIDTH;
	nSupportedImageSizes |= T30_SUPPORT_UNLIMITED_LENGTH;
	/*nSupportedImageSizes |= T30_SUPPORT_A4_LENGTH;
	nSupportedImageSizes |= T30_SUPPORT_B4_LENGTH;*/
	nSupportedImageSizes |= T30_SUPPORT_US_LETTER_LENGTH;
	nSupportedImageSizes |= T30_SUPPORT_US_LEGAL_LENGTH;

	/* Supported modems */
	nSupportedModems = 0;
	if ( nModem > 0 ) {
		nSupportedModems |= T30_SUPPORT_V27TER;
	}
	if ( nModem > 1 ) {
		nSupportedModems |= T30_SUPPORT_V29;
	}
	if ( nModem > 2 ) {
		nSupportedModems |= T30_SUPPORT_V17;
	}
	t30_set_supported_modems( psT30, nSupportedModems );

	/* Error correction */
	t30_set_ecm_capability( psT30, nEcm );

	t30_set_supported_compressions( psT30, nSupportedCompressions );
	t30_set_supported_resolutions( psT30, nSupportedResolutions );
	t30_set_supported_image_sizes( psT30, nSupportedImageSizes );

	/* spandsp loglevel */
	if ( nLogLevel >= 1 ) {
#if SPANDSP_VERSION >= 6
		psLogging = t30_get_logging_state( psT30 );
		span_log_set_level( psLogging, 0xFFFFFF );
#else
		psLogging = &psT30 -> psLogging;
		psLogging -> level = 0xFFFFFF;
#endif

		if ( logging == NULL ) {
			logging = spandsp_msg_log;
		}

		span_log_set_message_handler( psLogging, logging );
	}

#if SPANDSP_VERSION >= 5
	if ( pnLsi ) {
		t30_set_tx_ident( psT30, pnLsi );
	}
	if ( pnLocalHeaderInfo ) {
		t30_set_tx_page_header_info( psT30, pnLocalHeaderInfo );
	}
#endif

#if SPANDSP_VERSION == 4
	if ( pnLsi ) {
		t30_set_local_ident( psT30, pnLsi );
	}
	if ( pnLocalHeaderInfo ) {
		t30_set_header_info( psT30, pnLocalHeaderInfo );
	}
#endif

	if ( nSending == TRUE ) {
		t30_set_tx_file( psT30, pnTiffFile, -1, -1 );
	} else {
		t30_set_rx_file( psT30, pnTiffFile, -1 );
	}

	t30_set_phase_b_handler( psT30, phaseHandlerB, ( void * ) psConnection );
	t30_set_phase_d_handler( psT30, phaseHandlerD, ( void * ) psConnection );
	t30_set_phase_e_handler( psT30, phaseHandlerE, ( void * ) psConnection );

#if SPANDSP_VERSION >= 6
	t30_set_real_time_frame_handler( psT30, realTimeFrameHandler, ( void * ) psConnection );
#endif

	return 0;
}

/**
 * \brief Close spandsp
 * \param psFaxState fax state
 * \return error code
 */
int spandsp_close( fax_state_t *psFaxState ) {
	struct sFaxStatus *psStatus = NULL;
	struct sSession *psSession = faxophoneGetSession();
	int nI;

	if ( psFaxState != NULL ) {
		fax_release( psFaxState );
	} else {
		for ( nI = 0; nI < CAPI_CONNECTIONS; nI++ ) {
			psStatus = psSession -> asConnection[ nI ].pPrivate;

			if ( psStatus != NULL && psStatus -> psFaxState != NULL ) {
				fax_release( psStatus -> psFaxState );
			}
		}
	}

	return 0;
}

/**
 * \brief TX direction
 * \param psFaxState fax state
 * \param pnBuf transfer buffer
 * \param nLen length of buffer
 * \return error code
 */
int spandsp_tx( fax_state_t *psFaxState, uint8_t *pnBuf, size_t nLen ) {
	int16_t anBufIn[ CAPI_PACKETS ];
	uint8_t *pnAlaw;
	int nErr, nI;

	nErr = fax_tx( psFaxState, anBufIn, CAPI_PACKETS );
	pnAlaw = pnBuf;

	for ( nI = 0; nI != nLen; ++nI, ++pnAlaw ) {
		*pnAlaw = _linear16_2_law[ ( int16_t ) anBufIn[ nI ] ];
	}

	return nErr;
}

/**
 * \brief Process rx data through spandsp
 * \param psFaxState fax state information
 * \param pnBuf receive buffer
 * \param nLen length of buffer
 * \return error code
 */
int spandsp_rx( fax_state_t *psFaxState, uint8_t *pnBuf, size_t nLen ) {
	int16_t anBufIn[ CAPI_PACKETS ];
	int16_t *pnWave;
	int nErr, nI;

	pnWave = anBufIn;

	for ( nI = 0; nI != nLen; ++nI, ++pnWave ) {
		*pnWave = _law_2_linear16[ ( uint8_t ) pnBuf[ nI ] ];
	}

	nErr = fax_rx( psFaxState, anBufIn, CAPI_PACKETS );

	return nErr;
}

/**
 * \brief Receive/Transmit fax state
 * \param psConnection capi connection pointer
 * \param sCapiMessage current capi message
 */
void faxTransfer( struct sCapiConnection *psConnection, _cmsg sCapiMessage ) {
	struct sFaxStatus *psStatus = psConnection -> pPrivate;
	_cmsg sCmsg;
	uint8_t anAlawBufferRx[ CAPI_PACKETS ];
	uint8_t anAlawBufferTx[ CAPI_PACKETS ];
	int nLen = DATA_B3_IND_DATALENGTH( &sCapiMessage );
	int nRet;
	struct sSession *psSession = faxophoneGetSession();

	memset( anAlawBufferRx, 0, sizeof( anAlawBufferRx ) );
	memset( anAlawBufferTx, 0, sizeof( anAlawBufferTx ) );

	/* buffer-length check */
	if ( nLen > CAPI_PACKETS ) {
		FopDebug( FOP_DEBUG, "Warning: Buffer overflow! %d > %d\n", nLen, CAPI_PACKETS );
		nLen = CAPI_PACKETS;
	}

	/* Save data and send response */
	memcpy( anAlawBufferRx, ( uint8_t * ) DATA_B3_IND_DATA( &sCapiMessage ), nLen );
	isdnLock();
	DATA_B3_RESP( &sCmsg, psSession -> nApplId, psSession -> nMessageNumber++, psConnection -> nNcci, DATA_B3_IND_DATAHANDLE( &sCapiMessage ) );
	isdnUnlock();

	/* RX/TX spandsp */
	nRet = spandsp_rx( psStatus -> psFaxState, anAlawBufferRx, nLen );
	if ( !nRet ) {
		nRet = spandsp_tx( psStatus -> psFaxState, anAlawBufferTx, nLen );
	}

	/* Send data to remote */
	isdnLock();
	DATA_B3_REQ( &sCmsg, psSession -> nApplId, 0, psConnection -> nNcci, ( void * ) anAlawBufferTx, nLen, psSession -> nMessageNumber++, 0 );
	isdnUnlock();
}

/**
 * \brief Send Fax
 * \param pnTiffFile The Tiff file to send
 * \param nModem 0-3 (2400-14400)
 * \param nEcm Error correction mode (on/off)
 * \param nController The controller for sending the fax
 * \param pnSrcNo MSN
 * \param pnTrgNo Target fax number
 * \param pnLsi Fax ident
 * \param pnLocalHeaderInfo Fax header line
 * \param nCallAnonymous Send fax anonymous
 * \return error code
 */
struct sCapiConnection *faxSend( const char *pnTiffFile, int nModem, int nEcm, int nController, const char *pnSrcNo, const char *pnTrgNo, const char *pnLsi, const char *pnLocalHeaderInfo, int nCallAnonymous ) {
	struct sFaxStatus *psStatus = NULL;
	struct sCapiConnection *psConnection = NULL;
	int nCip = SPEECH_CIP;

	FopDebug( FOP_DEBUG, "tiff: %s, modem: %d, ecm: %s, controller: %d, src: %s, trg: %s, ident: %s, header: %s, anonymous: %d)\n", pnTiffFile, nModem, nEcm ? "on" : "off", nController, pnSrcNo, pnTrgNo, ( pnLsi != NULL ? pnLsi : "(null)"), ( pnLocalHeaderInfo != NULL ? pnLocalHeaderInfo : "(null)" ), nCallAnonymous );

	psStatus = malloc( sizeof( struct sFaxStatus ) );
	memset( psStatus, 0, sizeof( struct sFaxStatus ) );

	psStatus -> ePhase = IDLE;
	psStatus -> nErrorCode = 0;
	psStatus -> nSending = 1;
	psStatus -> nManualHookup = 0;
	psStatus -> nModem = nModem;
	psStatus -> nEcm = nEcm;
	snprintf( psStatus -> anHeader, sizeof( psStatus -> anHeader ), "%s", pnLocalHeaderInfo );
	snprintf( psStatus -> anIdent, sizeof( psStatus -> anIdent ), "%s", pnLsi );
	snprintf( psStatus -> anSrcNo, sizeof( psStatus -> anSrcNo ), "%s", pnSrcNo );
	snprintf( psStatus -> anTrgNo, sizeof( psStatus -> anTrgNo ), "%s", pnTrgNo );
	snprintf( psStatus -> anTiffFile, sizeof( psStatus -> anTiffFile ), "%s", pnTiffFile );

	if ( nController == 4 || nController == 5 ) {
		nCip = SPEECH_CIP;
		FopDebug( FOP_DEBUG, "Using 'Analog Fax' id\n" );
	} else {
		nCip = FAX_CIP;
		FopDebug( FOP_DEBUG, "Using 'ISDN Fax' id\n" );
	}
	psConnection = capiCall( nController, pnSrcNo, pnTrgNo, ( unsigned int ) nCallAnonymous, SESSION_FAX, nCip );
	if ( psConnection != NULL ) {
		psConnection -> pPrivate = psStatus;
		spandsp_init( psStatus -> anTiffFile, TRUE, psStatus -> nModem, psStatus -> nEcm, psStatus -> anIdent, psStatus -> anHeader, psConnection );
	}

	return psConnection;
}

/**
 * \brief Set fax debug level
 * \param nLevel debug level
 */
void faxSetLogLevel( int nLevel ) {
	nLogLevel = nLevel;
}

/**
 * \brief Set fax log handler
 * \param logFunc log function
 */
void faxSetLogHandler( void (*logFunc )( int nLevel, const char *pnText ) ) {
	logging = logFunc;
}

/**
 * \brief Receive Fax
 * \param pnTiffFile The Tiff file for saving the fax
 * \param nModem 0-3 (2400-14400)
 * \param nEcm Error correction mode (on/off)
 * \param pnSrcNo MSN
 * \param pnTrgNo After receiving a fax, dst_no is the senders fax number
 * \param nManualHookup: Hook up manually
 * \return error code
 */
int FaxReceive( struct sCapiConnection *psConnection, const char *pnTiffFile, int nModem, int nEcm, const char *pnSrcNo, char *pnTrgNo, int nManualHookup ) {
	struct sFaxStatus *psStatus = NULL;
	int nRet = -2;

	FopDebug( FOP_DEBUG, "tiff: %s, modem: %d, ecm: %s, src: %s, manual: %s)\n", pnTiffFile, nModem, nEcm ? "on" : "off", pnSrcNo, nManualHookup ? "on" : "off" );

	if ( psConnection != NULL ) {
		psStatus = malloc( sizeof( struct sFaxStatus ) );
		memset( psStatus, 0, sizeof( struct sFaxStatus ) );

		psStatus -> ePhase = IDLE;
		psStatus -> nSending = 0;
		psStatus -> nModem = nModem;
		psStatus -> nEcm = nEcm;
		psStatus -> nManualHookup = nManualHookup;
		psStatus -> nErrorCode = 0;

		snprintf( psStatus -> anSrcNo, sizeof( psStatus -> anSrcNo ), "%s", pnSrcNo );
		snprintf( psStatus -> anTiffFile, sizeof( psStatus -> anTiffFile ), "%s", pnTiffFile );

		psConnection -> pPrivate = psStatus;

		spandsp_init( psStatus -> anTiffFile, FALSE, psStatus -> nModem, psStatus -> nEcm, psStatus -> anIdent, psStatus -> anHeader, psConnection );

		snprintf( pnTrgNo, sizeof( psStatus -> anTrgNo ), "%s", psStatus -> anTrgNo );
	}

	return nRet;
}

/**
 * \brief Cleanup private fax structure from capi connection
 * \param psConnection capi connection
 */
void faxClean( struct sCapiConnection *psConnection ) {
	struct sFaxStatus *psStatus = psConnection -> pPrivate;

	FopDebug( FOP_DEBUG, "Sending phaseHandlerE with result -1\n");
	phaseHandlerE( fax_get_t30_state( psStatus -> psFaxState ), psConnection, -1 );

	spandsp_close( psStatus -> psFaxState );

	free( psStatus );
	psConnection -> pPrivate = NULL;
}
