/* $Id$ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __WAIT_STAT_HPP_INCLUDED
#define __WAIT_STAT_HPP_INCLUDED

#include <list>
#include "frontend/ast/ConditionedStat.hpp"
#include "frontend/ast/Name.hpp"

namespace ast {

//! a VHDL wait statement.
/** This class represents a VHDL wait statement. */
class WaitStat : public ConditionedStat {
public:
	//! c'tor
	/** @param sens Sensitivity list of signals.
          * @param cond Condition to stop waiting.
          * @param time wait for max timeout.
	  * @param loc location of the wait statement.
          */
	WaitStat(
		std::list<Name*> *sens, 
		Expression *cond, 
		Expression *time,
		Location loc
		) :	ConditionedStat(cond, loc),
			sensitivities(sens),
			timeout(time) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor &visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		stream << "WAIT";
		if (this->sensitivities) {
			stream << " ON ";
			util::MiscUtil::listPut(this->sensitivities, stream, ", ");
		}

		if (this->timeout) {
			stream << " FOR " << this->timeout;
		}

		stream << ';';
	}


	/** sensitivity list to wait on*/
	std::list<Name*> *sensitivities;
	/** wait maximal for timeout */
	Expression *timeout;

protected:
	/** Destructor */
	virtual ~WaitStat() {
		util::MiscUtil::lterminate(sensitivities);
		util::MiscUtil::terminate(timeout);
	}
};

}; /* namespace ast */

#endif /* __WAIT_STAT_HPP_INCLUDED */
