/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com */

package com.db4odoc.performance;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import com.db4o.Db4o;
import com.db4o.ObjectContainer;
import com.db4o.ObjectServer;
import com.db4o.config.Configuration;
import com.db4o.config.QueryEvaluationMode;
import com.db4o.query.Constraint;
import com.db4o.query.Predicate;
import com.db4o.query.Query;

public class QueryPerformanceBenchmark {

	private static int _count = 100000;

	private static int _depth = 3;

	private static boolean _isClientServer = false;

	private static boolean TCP = true;

	private static String _filePath = "performance.db4o";

	private static String _host = "localhost";

	private static final int PORT = 4477;

	private ObjectContainer objectContainer;

	private ObjectServer objectServer;

	private long startTime;

	public static void main(String[] arguments) {
//		new QueryPerformanceBenchmark().runDifferentQueriesTest();
//		new QueryPerformanceBenchmark().runDifferentObjectsTest();
//		new QueryPerformanceBenchmark().runRamDiskTest();
//		new QueryPerformanceBenchmark().runClientServerTest();
//		new QueryPerformanceBenchmark().runIndexTest();
//		new QueryPerformanceBenchmark().runInheritanceTest();
//	    new QueryPerformanceBenchmark().runQueryStructureTest();
		new QueryPerformanceBenchmark().runQueryAmountOfObjectsTest();
	}

	// end main

	private void runQueryAmountOfObjectsTest() {
		init();
		clean();
		System.out.println("Storing " + _count + " of  objects of depth " + _depth);
		open(configure());
		store();
		close();

		//
		open(configure());
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		Item item = (Item) query.execute().next();
		stopTimer("Select 1 object SODA: " + item._name);
		
		System.out.println("Add some objects of another type and check the query time again:");
		storeWithArray();
		close();
		//
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		item = (Item) query.execute().next();
		stopTimer("Select 1 object SODA: " + item._name);
		close();
		
		
		// Add many objects of the same type
		initLargeDb();
		clean();
		System.out.println("Storing " + _count + " of  objects of depth " + _depth);
		open(configure());
		store();
		close();

		//
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		item = (Item) query.execute().next();
		stopTimer("Select 1 object SODA: " + item._name);
		close();
	}
	// end runQueryAmountOfObjectsTest

	private void runQueryStructureTest() {
		init();

		clean();
		System.out.println("Storing objects as a bulk:");
		open(configure());
		store();
		close();

		//
		open(configure());
		System.out.println("Simple SODA query:");
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		Item item = (Item) query.execute().next();
		stopTimer("Select 1 object SODA: " + item._name);
		close();
		
		//
		open(configure());
		System.out.println("Sorted SODA query:");
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").orderDescending();
		item = (Item) query.execute().next();
		stopTimer("Select 1 object SODA: " + item._name);
		close();
		
		//
		open(configure());
		System.out.println("SODA query with joins:");
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		Constraint con = query.constrain("level2/1");
		query.descend("_name").orderDescending().constrain("level1/1").or(con);
		List result = query.execute();
		stopTimer("Selected " + result.size() + " objects SODA");
		close();
		
	}
	// end runQueryStructureTest

	private void runDifferentQueriesTest() {
		init();

		clean();
		System.out.println("Storing objects as a bulk:");
		open(configure());
		store();
		close();

		open(configure());
		//
		System.out.println("Query by example:");
		startTimer();
		Item item = (Item) objectContainer.queryByExample(
				new Item("level1/1", null)).next();
		stopTimer("Select 1 object QBE: " + item._name);

		//
		System.out.println("SODA:");
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		item = (Item) query.execute().next();
		stopTimer("Select 1 object SODA: " + item._name);

		//
		System.out.println("Native Query:");
		startTimer();
		List<Item> result = objectContainer.query(new Predicate<Item>() {
			public boolean match(Item item) {
				return item._name.equals("level1/1");
			}
		});
		item = result.get(0);
		stopTimer("Select 1 object NQ: " + item._name);
		close();
		
		//
		open(configureUnoptimizedNQ());
		System.out.println("Native Query Unoptimized:");
		startTimer();
		result = objectContainer.query(new Predicate<Item>() {
			public boolean match(Item item) {
				return item._name.equals("level1/1");
			}
		});
		item = result.get(0);
		stopTimer("Select 1 object NQ: " + item._name);

		close();
	}

	// end runDifferentQueriesTest

	
	private void runRamDiskTest() {

		initForHardDriveTest();
		clean();
		System.out.println("Storing " + _count + " objects of depth " + _depth
				+ " on a hard drive:");
		open(configureRamDrive());
		store();
		close();
		open(configureRamDrive());
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		Item item = (Item) query.execute().next();
		stopTimer("Select 1 object: " + item._name);
		close();

		initForRamDriveTest();
		clean();
		System.out.println("Storing " + _count + " objects of depth " + _depth
				+ " on a RAM disk:");
		open(configureRamDrive());
		store();
		close();
		open(configureRamDrive());
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		item = (Item) query.execute().next();
		stopTimer("Select 1 object: " + item._name);
		close();
	}

	// end runRamDiskTest

	private void runClientServerTest() {

		initForClientServer();
		clean();
		System.out.println("Storing " + _count + " objects of depth " + _depth
				+ " remotely:");
		open(configureClientServer());
		store();
		close();
		open(configureClientServer());
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		Item item = (Item) query.execute().next();
		stopTimer("Select 1 object: " + item._name);
		close();
		
		init();
		clean();
		System.out.println("Storing " + _count + " objects of depth " + _depth
				+ " locally:");
		open(configureClientServer());
		store();
		close();
		open(configureClientServer());
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		item = (Item) query.execute().next();
		stopTimer("Select 1 object: " + item._name);
		close();
	}

	// end runClientServerTest

	private void runInheritanceTest() {
		init();
		clean();
		System.out.println("Storing " + _count + " objects of depth " + _depth);
		open(configure());
		store();
		close();
		open(configure());
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		Item item = (Item) query.execute().next();
		stopTimer("Select 1 object: " + item._name);
		close();

		clean();
		System.out.println("Storing " + _count + " inherited objects of depth "
				+ _depth);
		open(configure());
		storeInherited();
		close();
		open(configure());
		startTimer();
		// Query for item, inheriting objects should be included in the result
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/1");
		item = (Item) query.execute().next();
		stopTimer("Select 1 object: " + item._name);
		close();
	}

	// end runInheritanceTest

	private void runDifferentObjectsTest() {

		init();
		System.out.println("Storing " + _count + " objects with " + _depth
				+ " levels of embedded objects:");

		clean();
		System.out.println();
		System.out.println(" - primitive object with int field");
		open(configure());
		storeSimplest();
		close();
		open(configure());
		startTimer();
		Query query = objectContainer.query();
		query.constrain(SimplestItem.class);
		query.descend("_id").constrain(1);
		List result = query.execute();
		SimplestItem simplestItem = (SimplestItem) result.get(0);
		stopTimer("Querying SimplestItem: " + simplestItem._id);
		close();

		open(configure());
		System.out.println();
		System.out.println(" - object with String field");
		store();
		close();
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/2");
		result = query.execute();
		Item item = (Item) result.get(0);
		stopTimer("Querying object with String field: " + item._name);
		close();

		clean();
		open(configure());
		System.out.println();
		System.out.println(" - object with StringBuffer field");
		storeWithStringBuffer();
		close();
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(ItemWithStringBuffer.class);
		query.descend("_name").constrain(new StringBuffer("level1/2"));
		result = query.execute();
		ItemWithStringBuffer itemWithSB = (ItemWithStringBuffer) result.get(0);
		stopTimer("Querying object with StringBuffer field: "
				+ itemWithSB._name);
		close();

		clean();
		open(configure());
		System.out.println();
		System.out.println(" - object with int array field");
		storeWithArray();
		close();
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(ItemWithArray.class);
		Query idQuery = query.descend("_id");
		idQuery.constrain(new Integer(1));
		idQuery.constrain(new Integer(2));
		idQuery.constrain(new Integer(3));
		idQuery.constrain(new Integer(4));
		result = query.execute();

		ItemWithArray itemWithArray = (ItemWithArray) result.get(0);
		stopTimer("Querying object with Array field: [" + itemWithArray._id[0]
				+ ", " + +itemWithArray._id[1] + ", " + +itemWithArray._id[2]
				+ ", " + +itemWithArray._id[0] + "]");
		close();

		clean();
		open(configure());
		System.out.println();
		System.out.println(" - object with ArrayList field");
		storeWithArrayList();
		close();
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(ItemWithArrayList.class);
		query.descend("_ids").constrain(1).contains();
		result = query.execute();
		ItemWithArrayList itemWithArrayList = (ItemWithArrayList) result.get(0);
		stopTimer("Querying object with ArrayList field: "
				+ itemWithArrayList._ids.toString());
		close();

	}

	// end runDifferentObjectsTest

	private void runIndexTest() {

		init();
		System.out.println("Storing " + _count + " objects with " + _depth
				+ " levels of embedded objects:");

		clean();
		System.out.println(" - no index");
		open(configure());
		store();
		close();
		open(configure());
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/2");
		List result = query.execute();
		Item item = (Item) result.get(0);
		stopTimer("Querying object with String field: " + item._name);
		close();


		System.out.println(" - index on String field");
		// open to create index
		open(configureIndex());
		close();
		open(configure());
		startTimer();
		query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level1/2");
		result = query.execute();
		item = (Item) result.get(0);
		stopTimer("Querying object with String field: " + item._name);
		close();
	}

	// end runIndexTest

	private void init() {
		_filePath = "performance.db4o";
		// amount of objects
		_count = 10000;
		// depth of objects
		_depth = 3;
		_isClientServer = false;

	}

	// end init

	private void initLargeDb() {
		_filePath = "performance.db4o";
		_count = 100000;
		_depth = 3;
		_isClientServer = false;

	}

	// end initLargeDb

	
	private void initForClientServer() {
		_filePath = "performance.db4o";
		_isClientServer = true;
		_host = "localhost";
	}

	// end initForClientServer

	private void initForRamDriveTest() {
		_count = 30000;
		_depth = 3;
		_filePath = "r:\\performance.db4o";
		_isClientServer = false;

	}

	// end initForRamDriveTest

	private void initForHardDriveTest() {
		_count = 30000;
		_depth = 3;
		_filePath = "performance.db4o";
		_isClientServer = false;

	}

	// end initForHardDriveTest


	private void clean() {
		new File(_filePath).delete();
	}

	// end clean

	private Configuration configure() {
		Configuration config = Db4o.newConfiguration();
		return config;
	}

	// end configure

	private Configuration configureUnoptimizedNQ() {
		Configuration config = Db4o.newConfiguration();
		config.optimizeNativeQueries(false);
		return config;
	}
	// end configureUnoptimizedNQ

	private Configuration configureIndex() {
		Configuration config = Db4o.newConfiguration();
		config.objectClass(Item.class).objectField("_name").indexed(true);
		return config;
	}

	// end configureIndex

	private Configuration configureClientServer() {
		Configuration config = Db4o.newConfiguration();
		config.queries().evaluationMode(QueryEvaluationMode.IMMEDIATE);
		config.clientServer().singleThreadedClient(true);
		return config;
	}

	// end configureClientServer

	private Configuration configureRamDrive() {
		Configuration config = Db4o.newConfiguration();
		config.flushFileBuffers(true);
		return config;
	}

	// end configureRamDrive

	private void store() {
		startTimer();
		for (int i = 0; i < _count; i++) {
			Item item = new Item("level" + i, null);
			for (int j = 1; j < _depth; j++) {
				item = new Item("level" + i + "/" + j, item);
			}
			objectContainer.store(item);
		}
		objectContainer.commit();
		stopTimer("Store " + totalObjects() + " objects");
	}

	// end store

	private void storeInherited() {
		startTimer();
		for (int i = 0; i < _count; i++) {
			ItemDerived item = new ItemDerived("level" + i, null);
			for (int j = 1; j < _depth; j++) {
				item = new ItemDerived("level" + i + "/" + j, item);
			}
			objectContainer.store(item);
		}
		objectContainer.commit();
		stopTimer("Store " + totalObjects() + " objects");
	}

	// end storeInherited

	private void storeWithStringBuffer() {
		startTimer();
		for (int i = 0; i < _count; i++) {
			ItemWithStringBuffer item = new ItemWithStringBuffer(
					new StringBuffer("level" + i), null);
			for (int j = 1; j < _depth; j++) {
				item = new ItemWithStringBuffer(new StringBuffer("level" + i
						+ "/" + j), item);
			}
			objectContainer.store(item);
		}
		objectContainer.commit();
		stopTimer("Store " + totalObjects() + " objects");
	}

	// end storeWithStringBuffer

	private void storeSimplest() {
		startTimer();
		for (int i = 0; i < _count; i++) {
			SimplestItem item = new SimplestItem(i, null);
			for (int j = 1; j < _depth; j++) {
				item = new SimplestItem(i, item);
			}
			objectContainer.store(item);
		}
		objectContainer.commit();
		stopTimer("Store " + totalObjects() + " objects");
	}

	// end storeSimplest

	private void storeWithArray() {
		startTimer();
		int[] array = new int[] { 1, 2, 3, 4 };
		for (int i = 0; i < _count; i++) {
			int[] id = new int[] { 1, 2, 3, 4 };
			ItemWithArray item = new ItemWithArray(id, null);
			for (int j = 1; j < _depth; j++) {
				int[] id1 = new int[] { 1, 2, 3, 4 };
				item = new ItemWithArray(id1, item);
			}
			objectContainer.store(item);
		}
		objectContainer.commit();
		stopTimer("Store " + totalObjects() + " objects");
	}

	// end storeWithArray

	private void storeWithArrayList() {
		startTimer();
		ArrayList idList = new ArrayList();
		idList.add(1);
		idList.add(2);
		idList.add(3);
		idList.add(4);
		for (int i = 0; i < _count; i++) {
			ArrayList ids = new ArrayList();
			ids.addAll(idList);
			ItemWithArrayList item = new ItemWithArrayList(ids, null);
			for (int j = 1; j < _depth; j++) {
				ArrayList ids1 = new ArrayList();
				ids1.addAll(idList);
				item = new ItemWithArrayList(ids1, item);
			}
			objectContainer.store(item);
		}
		objectContainer.commit();
		stopTimer("Store " + totalObjects() + " objects");
	}

	// end storeWithArrayList

	private int totalObjects() {
		return _count * _depth;
	}

	// end totalObjects

	private void open(Configuration configure) {
		if (_isClientServer) {
			int port = TCP ? PORT : 0;
			String user = "db4o";
			String password = user;
			objectServer = Db4o.openServer(configure, _filePath, port);
			objectServer.grantAccess(user, password);
			objectContainer = TCP ? Db4o
					.openClient(configure, _host, port, user, password) : objectServer
					.openClient(configure);
		} else {
			objectContainer = Db4o.openFile(configure, _filePath);
		}
	}

	// end open

	private void close() {
		objectContainer.close();
		if (_isClientServer) {
			objectServer.close();
		}
	}

	// end close

	private void startTimer() {
		startTime = System.currentTimeMillis();
	}

	// end startTimer

	private void stopTimer(String message) {
		long stop = System.currentTimeMillis();
		long duration = stop - startTime;
		System.out.println(message + ": " + duration + "ms");
	}

	// end stopTimer

	public static class Item {

		public String _name;
		public Item _child;

		public Item() {

		}

		public Item(String name, Item child) {
			_name = name;
			_child = child;
		}
	}

	// end Item

	public static class ItemDerived extends Item {

		public ItemDerived(String name, ItemDerived child) {
			super(name, child);
		}
	}

	// end ItemDerived

	public static class ItemWithStringBuffer {

		public StringBuffer _name;
		public ItemWithStringBuffer _child;

		public ItemWithStringBuffer() {
		}

		public ItemWithStringBuffer(StringBuffer name,
				ItemWithStringBuffer child) {
			_name = name;
			_child = child;
		}
	}

	// end ItemWithStringBuffer

	public static class SimplestItem {

		public int _id;
		public SimplestItem _child;

		public SimplestItem() {
		}

		public SimplestItem(int id, SimplestItem child) {
			_id = id;
			_child = child;
		}
	}

	// end SimplestItem

	public static class ItemWithArray {

		public int[] _id;
		public ItemWithArray _child;

		public ItemWithArray() {
		}

		public ItemWithArray(int[] id, ItemWithArray child) {
			_id = id;
			_child = child;
		}
	}

	// end ItemWithArray

	public static class ItemWithArrayList {

		public ArrayList _ids;
		public ItemWithArrayList _child;

		public ItemWithArrayList() {
		}

		public ItemWithArrayList(ArrayList ids, ItemWithArrayList child) {
			_ids = ids;
			_child = child;
		}
	}
	// end ItemWithArrayList
}
