#include <assert.h>
#include <errno.h>
#include <netinet/in.h>
#include <pthread.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/epoll.h>
#include <sys/sendfile.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#include <algorithm>
#include <map>
#include <string>
#include <utility>
#include <vector>

#include "accesslog.h"
#include "log.h"
#include "markpool.h"
#include "metacube2.h"
#include "mutexlock.h"
#include "parse.h"
#include "server.h"
#include "state.pb.h"
#include "stream.h"
#include "util.h"

#ifndef SO_MAX_PACING_RATE
#define SO_MAX_PACING_RATE 47
#endif

using namespace std;

extern AccessLogThread *access_log;

Server::Server()
{
	pthread_mutex_init(&mutex, NULL);
	pthread_mutex_init(&queued_clients_mutex, NULL);

	epoll_fd = epoll_create(1024);  // Size argument is ignored.
	if (epoll_fd == -1) {
		log_perror("epoll_fd");
		exit(1);
	}
}

Server::~Server()
{
	for (size_t i = 0; i < streams.size(); ++i) {	
		delete streams[i];
	}

	safe_close(epoll_fd);
}

vector<ClientStats> Server::get_client_stats() const
{
	vector<ClientStats> ret;

	MutexLock lock(&mutex);
	for (map<int, Client>::const_iterator client_it = clients.begin();
	     client_it != clients.end();
	     ++client_it) {
		ret.push_back(client_it->second.get_stats());
	}
	return ret;
}

void Server::do_work()
{
	while (!should_stop()) {
		// Wait until there's activity on at least one of the fds,
		// or 20 ms (about one frame at 50 fps) has elapsed.
		//
		// We could in theory wait forever and rely on wakeup()
		// from add_client_deferred() and add_data_deferred(),
		// but wakeup is a pretty expensive operation, and the
		// two threads might end up fighting over a lock, so it's
		// seemingly (much) more efficient to just have a timeout here.
		int nfds = epoll_pwait(epoll_fd, events, EPOLL_MAX_EVENTS, EPOLL_TIMEOUT_MS, &sigset_without_usr1_block);
		if (nfds == -1 && errno != EINTR) {
			log_perror("epoll_wait");
			exit(1);
		}

		MutexLock lock(&mutex);  // We release the mutex between iterations.
	
		process_queued_data();

		for (int i = 0; i < nfds; ++i) {
			Client *client = reinterpret_cast<Client *>(events[i].data.u64);

			if (events[i].events & (EPOLLERR | EPOLLRDHUP | EPOLLHUP)) {
				close_client(client);
				continue;
			}

			process_client(client);
		}

		for (size_t i = 0; i < streams.size(); ++i) {	
			vector<Client *> to_process;
			swap(streams[i]->to_process, to_process);
			for (size_t i = 0; i < to_process.size(); ++i) {
				process_client(to_process[i]);
			}
		}
	}
}

CubemapStateProto Server::serialize()
{
	// We don't serialize anything queued, so empty the queues.
	process_queued_data();

	// Set all clients in a consistent state before serializing
	// (ie., they have no remaining lost data). Otherwise, increasing
	// the backlog could take clients into a newly valid area of the backlog,
	// sending a stream of zeros instead of skipping the data as it should.
	//
	// TODO: Do this when clients are added back from serialized state instead;
	// it would probably be less wasteful.
	for (map<int, Client>::iterator client_it = clients.begin();
	     client_it != clients.end();
	     ++client_it) {
		skip_lost_data(&client_it->second);
	}

	CubemapStateProto serialized;
	for (map<int, Client>::const_iterator client_it = clients.begin();
	     client_it != clients.end();
	     ++client_it) {
		serialized.add_clients()->MergeFrom(client_it->second.serialize());
	}
	for (size_t i = 0; i < streams.size(); ++i) {	
		serialized.add_streams()->MergeFrom(streams[i]->serialize());
	}
	return serialized;
}

void Server::add_client_deferred(int sock)
{
	MutexLock lock(&queued_clients_mutex);
	queued_add_clients.push_back(sock);
}

void Server::add_client(int sock)
{
	pair<map<int, Client>::iterator, bool> ret =
		clients.insert(make_pair(sock, Client(sock)));
	assert(ret.second == true);  // Should not already exist.
	Client *client_ptr = &ret.first->second;

	// Start listening on data from this socket.
	epoll_event ev;
	ev.events = EPOLLIN | EPOLLET | EPOLLRDHUP;
	ev.data.u64 = reinterpret_cast<uint64_t>(client_ptr);
	if (epoll_ctl(epoll_fd, EPOLL_CTL_ADD, sock, &ev) == -1) {
		log_perror("epoll_ctl(EPOLL_CTL_ADD)");
		exit(1);
	}

	process_client(client_ptr);
}

void Server::add_client_from_serialized(const ClientProto &client)
{
	MutexLock lock(&mutex);
	Stream *stream;
	int stream_index = lookup_stream_by_url(client.url());
	if (stream_index == -1) {
		assert(client.state() != Client::SENDING_DATA);
		stream = NULL;
	} else {
		stream = streams[stream_index];
	}
	pair<map<int, Client>::iterator, bool> ret =
		clients.insert(make_pair(client.sock(), Client(client, stream)));
	assert(ret.second == true);  // Should not already exist.
	Client *client_ptr = &ret.first->second;

	// Start listening on data from this socket.
	epoll_event ev;
	if (client.state() == Client::READING_REQUEST) {
		ev.events = EPOLLIN | EPOLLET | EPOLLRDHUP;
	} else {
		// If we don't have more data for this client, we'll be putting it into
		// the sleeping array again soon.
		ev.events = EPOLLOUT | EPOLLET | EPOLLRDHUP;
	}
	ev.data.u64 = reinterpret_cast<uint64_t>(client_ptr);
	if (epoll_ctl(epoll_fd, EPOLL_CTL_ADD, client.sock(), &ev) == -1) {
		log_perror("epoll_ctl(EPOLL_CTL_ADD)");
		exit(1);
	}

	if (client_ptr->state == Client::WAITING_FOR_KEYFRAME ||
	    (client_ptr->state == Client::SENDING_DATA &&
	     client_ptr->stream_pos == client_ptr->stream->bytes_received)) {
		client_ptr->stream->put_client_to_sleep(client_ptr);
	} else {
		process_client(client_ptr);
	}
}

int Server::lookup_stream_by_url(const std::string &url) const
{
	map<string, int>::const_iterator url_it = url_map.find(url);
	if (url_it == url_map.end()) {
		return -1;
	}
	return url_it->second;
}

int Server::add_stream(const string &url, size_t backlog_size, Stream::Encoding encoding)
{
	MutexLock lock(&mutex);
	url_map.insert(make_pair(url, streams.size()));
	streams.push_back(new Stream(url, backlog_size, encoding));
	return streams.size() - 1;
}

int Server::add_stream_from_serialized(const StreamProto &stream, int data_fd)
{
	MutexLock lock(&mutex);
	url_map.insert(make_pair(stream.url(), streams.size()));
	streams.push_back(new Stream(stream, data_fd));
	return streams.size() - 1;
}
	
void Server::set_backlog_size(int stream_index, size_t new_size)
{
	MutexLock lock(&mutex);
	assert(stream_index >= 0 && stream_index < ssize_t(streams.size()));
	streams[stream_index]->set_backlog_size(new_size);
}
	
void Server::set_encoding(int stream_index, Stream::Encoding encoding)
{
	MutexLock lock(&mutex);
	assert(stream_index >= 0 && stream_index < ssize_t(streams.size()));
	streams[stream_index]->encoding = encoding;
}
	
void Server::set_header(int stream_index, const string &http_header, const string &stream_header)
{
	MutexLock lock(&mutex);
	assert(stream_index >= 0 && stream_index < ssize_t(streams.size()));
	streams[stream_index]->http_header = http_header;
	streams[stream_index]->stream_header = stream_header;
}
	
void Server::set_mark_pool(int stream_index, MarkPool *mark_pool)
{
	MutexLock lock(&mutex);
	assert(clients.empty());
	assert(stream_index >= 0 && stream_index < ssize_t(streams.size()));
	streams[stream_index]->mark_pool = mark_pool;
}

void Server::set_pacing_rate(int stream_index, uint32_t pacing_rate)
{
	MutexLock lock(&mutex);
	assert(clients.empty());
	assert(stream_index >= 0 && stream_index < ssize_t(streams.size()));
	streams[stream_index]->pacing_rate = pacing_rate;
}

void Server::add_data_deferred(int stream_index, const char *data, size_t bytes, StreamStartSuitability suitable_for_stream_start)
{
	assert(stream_index >= 0 && stream_index < ssize_t(streams.size()));
	streams[stream_index]->add_data_deferred(data, bytes, suitable_for_stream_start);
}

// See the .h file for postconditions after this function.	
void Server::process_client(Client *client)
{
	switch (client->state) {
	case Client::READING_REQUEST: {
read_request_again:
		// Try to read more of the request.
		char buf[1024];
		int ret;
		do {
			ret = read(client->sock, buf, sizeof(buf));
		} while (ret == -1 && errno == EINTR);

		if (ret == -1 && errno == EAGAIN) {
			// No more data right now. Nothing to do.
			// This is postcondition #2.
			return;
		}
		if (ret == -1) {
			log_perror("read");
			close_client(client);
			return;
		}
		if (ret == 0) {
			// OK, the socket is closed.
			close_client(client);
			return;
		}

		RequestParseStatus status = wait_for_double_newline(&client->request, buf, ret);
	
		switch (status) {
		case RP_OUT_OF_SPACE:
			log(WARNING, "[%s] Client sent overlong request!", client->remote_addr.c_str());
			close_client(client);
			return;
		case RP_NOT_FINISHED_YET:
			// OK, we don't have the entire header yet. Fine; we'll get it later.
			// See if there's more data for us.
			goto read_request_again;
		case RP_EXTRA_DATA:
			log(WARNING, "[%s] Junk data after request!", client->remote_addr.c_str());
			close_client(client);
			return;
		case RP_FINISHED:
			break;
		}

		assert(status == RP_FINISHED);

		int error_code = parse_request(client);
		if (error_code == 200) {
			construct_header(client);
		} else {
			construct_error(client, error_code);
		}

		// We've changed states, so fall through.
		assert(client->state == Client::SENDING_ERROR ||
		       client->state == Client::SENDING_HEADER);
	}
	case Client::SENDING_ERROR:
	case Client::SENDING_HEADER: {
sending_header_or_error_again:
		int ret;
		do {
			ret = write(client->sock,
		                    client->header_or_error.data() + client->header_or_error_bytes_sent,
				    client->header_or_error.size() - client->header_or_error_bytes_sent);
		} while (ret == -1 && errno == EINTR);

		if (ret == -1 && errno == EAGAIN) {
			// We're out of socket space, so now we're at the “low edge” of epoll's
			// edge triggering. epoll will tell us when there is more room, so for now,
			// just return.
			// This is postcondition #4.
			return;
		}

		if (ret == -1) {
			// Error! Postcondition #1.
			log_perror("write");
			close_client(client);
			return;
		}
		
		client->header_or_error_bytes_sent += ret;
		assert(client->header_or_error_bytes_sent <= client->header_or_error.size());

		if (client->header_or_error_bytes_sent < client->header_or_error.size()) {
			// We haven't sent all yet. Fine; go another round.
			goto sending_header_or_error_again;
		}

		// We're done sending the header or error! Clear it to release some memory.
		client->header_or_error.clear();

		if (client->state == Client::SENDING_ERROR) {
			// We're done sending the error, so now close.	
			// This is postcondition #1.
			close_client(client);
			return;
		}

		// Start sending from the first keyframe we get. In other
		// words, we won't send any of the backlog, but we'll start
		// sending immediately as we get the next keyframe block.
		// This is postcondition #3.
		if (client->stream_pos == size_t(-2)) {
			client->stream_pos = std::min<size_t>(
			    client->stream->bytes_received - client->stream->backlog_size,
			    0);
			client->state = Client::SENDING_DATA;
		} else {
			// client->stream_pos should be -1, but it might not be,
			// if we have clients from an older version.
			client->stream_pos = client->stream->bytes_received;
			client->state = Client::WAITING_FOR_KEYFRAME;
		}
		client->stream->put_client_to_sleep(client);
		return;
	}
	case Client::WAITING_FOR_KEYFRAME: {
		Stream *stream = client->stream;
		if (ssize_t(client->stream_pos) > stream->last_suitable_starting_point) {
			// We haven't received a keyframe since this stream started waiting,
			// so keep on waiting for one.
			// This is postcondition #3.
			stream->put_client_to_sleep(client);
			return;
		}
		client->stream_pos = stream->last_suitable_starting_point;
		client->state = Client::SENDING_DATA;
		// Fall through.
	}
	case Client::SENDING_DATA: {
		skip_lost_data(client);
		Stream *stream = client->stream;

sending_data_again:
		size_t bytes_to_send = stream->bytes_received - client->stream_pos;
		assert(bytes_to_send <= stream->backlog_size);
		if (bytes_to_send == 0) {
			return;
		}

		// See if we need to split across the circular buffer.
		bool more_data = false;
		if ((client->stream_pos % stream->backlog_size) + bytes_to_send > stream->backlog_size) {
			bytes_to_send = stream->backlog_size - (client->stream_pos % stream->backlog_size);
			more_data = true;
		}

		ssize_t ret;
		do {
			off_t offset = client->stream_pos % stream->backlog_size;
			ret = sendfile(client->sock, stream->data_fd, &offset, bytes_to_send);
		} while (ret == -1 && errno == EINTR);

		if (ret == -1 && errno == EAGAIN) {
			// We're out of socket space, so return; epoll will wake us up
			// when there is more room.
			// This is postcondition #4.
			return;
		}
		if (ret == -1) {
			// Error, close; postcondition #1.
			log_perror("sendfile");
			close_client(client);
			return;
		}
		client->stream_pos += ret;
		client->bytes_sent += ret;

		if (client->stream_pos == stream->bytes_received) {
			// We don't have any more data for this client, so put it to sleep.
			// This is postcondition #3.
			stream->put_client_to_sleep(client);
		} else if (more_data && size_t(ret) == bytes_to_send) {
			goto sending_data_again;
		}
		break;
	}
	default:
		assert(false);
	}
}

// See if there's some data we've lost. Ideally, we should drop to a block boundary,
// but resync will be the mux's problem.
void Server::skip_lost_data(Client *client)
{
	Stream *stream = client->stream;
	if (stream == NULL) {
		return;
	}
	size_t bytes_to_send = stream->bytes_received - client->stream_pos;
	if (bytes_to_send > stream->backlog_size) {
		size_t bytes_lost = bytes_to_send - stream->backlog_size;
		client->stream_pos = stream->bytes_received - stream->backlog_size;
		client->bytes_lost += bytes_lost;
		++client->num_loss_events;

		double loss_fraction = double(client->bytes_lost) / double(client->bytes_lost + client->bytes_sent);
		log(WARNING, "[%s] Client lost %lld bytes (total loss: %.2f%%), maybe too slow connection",
			client->remote_addr.c_str(),
			(long long int)(bytes_lost),
			100.0 * loss_fraction);
	}
}

int Server::parse_request(Client *client)
{
	vector<string> lines = split_lines(client->request);
	if (lines.empty()) {
		return 400;  // Bad request (empty).
	}

	vector<string> request_tokens = split_tokens(lines[0]);
	if (request_tokens.size() < 2) {
		return 400;  // Bad request (empty).
	}
	if (request_tokens[0] != "GET") {
		return 400;  // Should maybe be 405 instead?
	}

	string url = request_tokens[1];
	if (url.find("?backlog") == url.size() - 8) {
		client->stream_pos = -2;
		url = url.substr(0, url.size() - 8);
	} else {
		client->stream_pos = -1;
	}

	map<string, int>::const_iterator url_map_it = url_map.find(url);
	if (url_map_it == url_map.end()) {
		return 404;  // Not found.
	}

	Stream *stream = streams[url_map_it->second];
	if (stream->http_header.empty()) {
		return 503;  // Service unavailable.
	}

	client->url = request_tokens[1];
	client->stream = stream;
	if (client->stream->mark_pool != NULL) {
		client->fwmark = client->stream->mark_pool->get_mark();
	} else {
		client->fwmark = 0;  // No mark.
	}
	if (setsockopt(client->sock, SOL_SOCKET, SO_MARK, &client->fwmark, sizeof(client->fwmark)) == -1) {                          
		if (client->fwmark != 0) {
			log_perror("setsockopt(SO_MARK)");
		}
	}
	if (setsockopt(client->sock, SOL_SOCKET, SO_MAX_PACING_RATE, &client->stream->pacing_rate, sizeof(client->stream->pacing_rate)) == -1) {
		if (client->stream->pacing_rate != ~0U) {
			log_perror("setsockopt(SO_MAX_PACING_RATE)");
		}
	}
	client->request.clear();

	return 200;  // OK!
}

void Server::construct_header(Client *client)
{
	Stream *stream = client->stream;
	if (stream->encoding == Stream::STREAM_ENCODING_RAW) {
		client->header_or_error = stream->http_header +
			"\r\n" +
			stream->stream_header;
	} else if (stream->encoding == Stream::STREAM_ENCODING_METACUBE) {
		client->header_or_error = stream->http_header +
			"Content-encoding: metacube\r\n" +
			"\r\n";
		if (!stream->stream_header.empty()) {
			metacube2_block_header hdr;
			memcpy(hdr.sync, METACUBE2_SYNC, sizeof(hdr.sync));
			hdr.size = htonl(stream->stream_header.size());
			hdr.flags = htons(METACUBE_FLAGS_HEADER);
			hdr.csum = htons(metacube2_compute_crc(&hdr));
			client->header_or_error.append(
				string(reinterpret_cast<char *>(&hdr), sizeof(hdr)));
		}
		client->header_or_error.append(stream->stream_header);
	} else {
		assert(false);
	}

	// Switch states.
	client->state = Client::SENDING_HEADER;

	epoll_event ev;
	ev.events = EPOLLOUT | EPOLLET | EPOLLRDHUP;
	ev.data.u64 = reinterpret_cast<uint64_t>(client);

	if (epoll_ctl(epoll_fd, EPOLL_CTL_MOD, client->sock, &ev) == -1) {
		log_perror("epoll_ctl(EPOLL_CTL_MOD)");
		exit(1);
	}
}
	
void Server::construct_error(Client *client, int error_code)
{
	char error[256];
	snprintf(error, 256, "HTTP/1.0 %d Error\r\nContent-type: text/plain\r\n\r\nSomething went wrong. Sorry.\r\n",
		error_code);
	client->header_or_error = error;

	// Switch states.
	client->state = Client::SENDING_ERROR;

	epoll_event ev;
	ev.events = EPOLLOUT | EPOLLET | EPOLLRDHUP;
	ev.data.u64 = reinterpret_cast<uint64_t>(client);

	if (epoll_ctl(epoll_fd, EPOLL_CTL_MOD, client->sock, &ev) == -1) {
		log_perror("epoll_ctl(EPOLL_CTL_MOD)");
		exit(1);
	}
}

template<class T>
void delete_from(vector<T> *v, T elem)
{
	typename vector<T>::iterator new_end = remove(v->begin(), v->end(), elem);
	v->erase(new_end, v->end());
}
	
void Server::close_client(Client *client)
{
	if (epoll_ctl(epoll_fd, EPOLL_CTL_DEL, client->sock, NULL) == -1) {
		log_perror("epoll_ctl(EPOLL_CTL_DEL)");
		exit(1);
	}

	// This client could be sleeping, so we'll need to fix that. (Argh, O(n).)
	if (client->stream != NULL) {
		delete_from(&client->stream->sleeping_clients, client);
		delete_from(&client->stream->to_process, client);
		if (client->stream->mark_pool != NULL) {
			int fwmark = client->fwmark;
			client->stream->mark_pool->release_mark(fwmark);
		}
	}

	// Log to access_log.
	access_log->write(client->get_stats());

	// Bye-bye!
	safe_close(client->sock);

	clients.erase(client->sock);
}
	
void Server::process_queued_data()
{
	{
		MutexLock lock(&queued_clients_mutex);

		for (size_t i = 0; i < queued_add_clients.size(); ++i) {
			add_client(queued_add_clients[i]);
		}
		queued_add_clients.clear();
	}

	for (size_t i = 0; i < streams.size(); ++i) {	
		streams[i]->process_queued_data();
	}
}
