# 
# Sofa support in MML
# 
# Note that only MML_SOFA_STABLE is currently/actively supported
#
# This file defines the following variables:
# - SOFA_SUPPORTED_VERSION      the name of the currently supported Sofa stable version
#                               this is equal to the git tag
# - SOFA_BUILD_DIR              To be set by the user to the Sofa build directory
#                               This is mandatory to have the Sofa support
#
# If Sofa support is enabled by the user by setting the build directory,
# the following variables are defined:
# - SOFA_SUPPORT            Set to ON
# - SOFA_SOURCE_DIR         The top-level directory of the Sofa source.
#                           Automatically extracted/hacked from the CMakeCache.txt in the
#                           build directory (${SOFA_BUILD_DIR})
# - SOFA_LIBS               List of Sofa libraries to link against needed at link time
# - SOFA_LIB_DIR            Directory where to find the compiled Sofa libraries
# - SOFA_INCLUDE_DIR        Sofa include directories needed a compile time
#
# If Sofa support is enabled by the user by setting the build directory,
# the following flags needs to be defined at compilation time:
# -DSOFA_SUPPORT
# -DSOFA_PATH="${SOFA_BUILD_DIR}"
# -DSOFA_STABLE

# list of possibly supported sofa STABLE version
# Only stable version are supported in MML
set(SOFA_STABLE_V15_03 "sofa-v15.03")

# The currently supported Sofa version 
# Only stable version are supported in MML
# To support a new version, add a new SOFA_STABLE_V... above and modify this
set(SOFA_SUPPORTED_VERSION ${SOFA_STABLE_V15_03})
  
# "find" SOFA (you need to have a SOFA_BUILD_DIR environment variable
# or to define it manually during configuration process)
set(SOFA_BUILD_DIR $ENV{SOFA_BUILD_DIR} CACHE PATH "Where to find SOFA (stable) ${SOFA_SUPPORTED_VERSION} build directory")
if(EXISTS ${SOFA_BUILD_DIR})
    # Compile SOFA .scn import filter? (ON or OFF)
    option(SOFA_SUPPORT "Compile SOFA MML Simulator?" ON)
endif()

# reset it
unset(SOFA_LIBS)

# sofa settings
if(SOFA_SUPPORT)
    # supported stable version 15.03
    if (SOFA_SUPPORTED_VERSION STREQUAL "${SOFA_STABLE_V15_03}")
        message(STATUS "SOFA support on (supporting ${SOFA_SUPPORTED_VERSION})")
        
        # TODO one day when Sofa includes an install phase and FindSofa.cmake, everything should be easier:
        #     # use Find SOFA script
        #     set(SOFA_BUILD_DIR ${SOFA_SOURCE_DIR})
        #     include(${SOFA_BUILD_DIR}/cmake/Modules/FindSOFA.cmake)
        #     list(APPEND CMAKE_MODULE_PATH "${SOFA_BUILD_DIR}/cmake/Modules")
        #     find_package(SOFA) 
        #     # Check the FindSofa.cmake and "translate" all their variables here
        #     set(SOFA_INCLUDE_DIR ${SOFA_INCLUDE_DIRS})
        #         set(SOFA_LIBS ${SOFA_LIB_CORE}
        #         ${SOFA_LIB_HELPER}
        #         ${SOFA_LIB_DEFAULT_TYPE})

        # For now, lets trick it!
        # Problem with the current way Sofa uses cmake:
        # - there is no install procedure
        # - everything needed at MML compile-time (headers mainly) are to be found in the source directory
        # - everything needed at MML link-time (lib mainly) are to be found in the build directory
        # The hack to solve this consists in parsing the CMakeCache file (which is in the build directory toplevel)
        # to automatically get the value the source directory (in the CMAKE_SOURCE_DIR variable)
        # So here it is in two steps 
        # 1. Get the correct line from the CMakeCache file 
        file(STRINGS ${SOFA_BUILD_DIR}/CMakeCache.txt SOFA_SOURCE_DIR_LINE REGEX "^Sofa_SOURCE_DIR:STATIC=(.*)$")
        # 2. Use a regex to extract the path from the line
        string(REGEX REPLACE "^Sofa_SOURCE_DIR:STATIC=(.*)$" "\\1" SOFA_SOURCE_DIR "${SOFA_SOURCE_DIR_LINE}")
        message(STATUS "Detected Sofa ${MML_SOFA_STABLE} source directory is ${SOFA_SOURCE_DIR}")

        # fill the list of needed libraries
        # the only way to find the required lib is to hack the files generated by cmake:
        # cd ${SOFA_BUILD_DIR}/applications/projects/runSofa
        # cat CMakeFiles/runSofa.dir/link.txt | sed 's/\s/\n/g' | grep ../../../lib | sed "s+../../../lib/lib++g" | sed "s+.so++g"
        # explained:
        # - first sed to explode the one line into one word per line
        # - first grep to get only the Sofa library
        # - second sed to remove the ../../../lib/ 
        # - third sed to remove the trailing .so
        # 
        set(SOFA_LIBS 
                SofaSimulationTree
                SofaGuiMain
                SofaComponentMain
                SofaSimulationGraph
                SofaGuiQt
                Qwt
                SofaGuiGlut
                SofaGuiCommon
                SofaComponentMain
                SofaComponentBase
                SofaBaseAnimationLoop
                SofaComponentCommon
                SofaComponentGeneral
                SofaExporter
                SofaValidation
                SofaLoader
                SofaEngine
                SofaHaptics
                SofaPreconditioner
                SofaComponentAdvanced
                SofaEulerianFluid
                SofaComponentMisc
                SofaMiscForceField
                SofaMiscFem
                SofaMiscEngine
                SofaNonUniformFem
                SofaMiscCollision
                SofaExplicitOdeSolver
                SofaConstraint
                SofaImplicitOdeSolver
                SofaVolumetricData
                SofaMiscSolver
                SofaMisc
                SofaMiscTopology
                SofaUserInteraction
                SofaGraphComponent
                SofaMeshCollision
                SofaBaseCollision
                SofaSphFluid
                SofaObjectInteraction
                SofaDeformable
                SofaMiscMapping
                SofaRigid
                SofaBaseMechanics
                SofaSimpleFem
                SofaOpenglVisual
                SofaTopologyMapping
                SofaBoundaryCondition
                SofaEigen2Solver
                SofaDenseSolver
                SofaBaseLinearSolver
                SofaBaseVisual
                SofaBaseTopology
                SofaSimulationTree
                SofaSimulationCommon
                SofaCore
                SofaDefaultType
                SofaHelper
                tinyxml
        )
        
        # Library path
        set(SOFA_LIB_DIR ${SOFA_BUILD_DIR}/lib)

        # fill in the required include path
        # use sofa source directory for sofa stable version
        set(SOFA_INCLUDE_DIR ${SOFA_SOURCE_DIR}/include 
                             ${SOFA_SOURCE_DIR}/modules 
                             ${SOFA_SOURCE_DIR}/framework
                             ${SOFA_SOURCE_DIR}/extlibs/miniBoost
                             ${SOFA_SOURCE_DIR}/include/modules
                             ${SOFA_SOURCE_DIR}/include/framework
                             ${SOFA_SOURCE_DIR}/include/extlibs/miniBoost
                             ${SOFA_SOURCE_DIR}/extlibs/tinyxml 
        )

    endif()
endif()
