/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 2006-2012, Digium, Inc.
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*! \file
 *
 * SHELL function to return the output generated by a command issued to the system shell.
 *
 * \note Inspiration and Guidance from Russell! Thank You!
 *
 * \author Brandon Kruse <bkruse@digium.com>
 *
 * \ingroup functions
 */

/*** MODULEINFO
	<support_level>core</support_level>
 ***/

#include "asterisk.h"

#include "asterisk/module.h"
#include "asterisk/channel.h"
#include "asterisk/pbx.h"
#include "asterisk/utils.h"
#include "asterisk/app.h"

static int shell_helper(struct ast_channel *chan, const char *cmd, char *data,
		                         char *buf, size_t len)
{
	int res = 0;

	if (ast_strlen_zero(data)) {
		ast_log(LOG_WARNING, "Missing Argument!  Example:  Set(foo=${SHELL(echo \"bar\")})\n");
		return -1;
	}

	if (chan) {
		ast_autoservice_start(chan);
	}

	if (len >= 1) {
		FILE *ptr;
		char plbuff[4096];

		ptr = popen(data, "r");
		if (ptr) {
			while (fgets(plbuff, sizeof(plbuff), ptr)) {
				strncat(buf, plbuff, len - strlen(buf) - 1);
			}
			pclose(ptr);
		} else {
			ast_log(LOG_WARNING, "Failed to execute shell command '%s'\n", data);
			res = -1;
		}
	}

	if (chan) {
		ast_autoservice_stop(chan);
	}

	return res;
}

/*** DOCUMENTATION
	<function name="SHELL" language="en_US">
		<since><version>1.6.2.0</version></since>
		<synopsis>
			Executes a command using the system shell and captures its output.
		</synopsis>
		<syntax>
			<parameter name="command" required="true">
				<para>The command that the shell should execute.</para>
				<warning><para>Do not use untrusted strings such as <variable>CALLERID(num)</variable>
				or <variable>CALLERID(name)</variable> as part of the command parameters.  You
				risk a command injection attack executing arbitrary commands if the untrusted
				strings aren't filtered to remove dangerous characters.  See function
				<variable>FILTER()</variable>.</para></warning>
			</parameter>
		</syntax>
		<description>
			<para>Collects the output generated by a command executed by the system shell</para>
			<example title="Shell example">
			exten => s,1,Set(foo=${SHELL(echo bar)})
			</example>
			<note>
				<para>The command supplied to this function will be executed by the
				system's shell, typically specified in the SHELL environment variable. There
				are many different system shells available with somewhat different behaviors,
				so the output generated by this function may vary between platforms.</para>

				<para>If <literal>live_dangerously</literal> in <literal>asterisk.conf</literal>
				is set to <literal>no</literal>, this function can only be executed from the
				dialplan, and not directly from external protocols.</para>
			</note>
		</description>

	</function>
 ***/
static struct ast_custom_function shell_function = {
	.name = "SHELL",
	.read = shell_helper,
};

static int unload_module(void)
{
	return ast_custom_function_unregister(&shell_function);
}

static int load_module(void)
{
	return ast_custom_function_register_escalating(&shell_function, AST_CFE_READ);
}

AST_MODULE_INFO_STANDARD(ASTERISK_GPL_KEY, "Collects the output generated by a command executed by the system shell");
