/*
 * Copyright (C) 2012 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

FILE_LICENCE ( GPL2_OR_LATER );

/** @file
 *
 * Big integer self-tests
 *
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <assert.h>
#include <string.h>
#include <ipxe/bigint.h>
#include <ipxe/test.h>

/** Define inline big integer */
#define BIGINT(...) { __VA_ARGS__ }

/* Provide global functions to allow inspection of generated assembly code */

void bigint_init_sample ( bigint_element_t *value0, unsigned int size,
			  const void *data, size_t len ) {
	bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( void * ) value0 );

	bigint_init ( value, data, len );
}

void bigint_done_sample ( const bigint_element_t *value0, unsigned int size,
			  void *out, size_t len ) {
	const bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( const void * ) value0 );

	bigint_done ( value, out, len );
}

void bigint_add_sample ( const bigint_element_t *addend0,
			 bigint_element_t *value0, unsigned int size ) {
	const bigint_t ( size ) *addend __attribute__ (( may_alias ))
		= ( ( const void * ) addend0 );
	bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( void * ) value0 );

	bigint_add ( addend, value );
}

void bigint_subtract_sample ( const bigint_element_t *subtrahend0,
			      bigint_element_t *value0, unsigned int size ) {
	const bigint_t ( size ) *subtrahend __attribute__ (( may_alias ))
		= ( ( const void * ) subtrahend0 );
	bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( void * ) value0 );

	bigint_subtract ( subtrahend, value );
}

void bigint_rol_sample ( bigint_element_t *value0, unsigned int size ) {
	bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( void * ) value0 );

	bigint_rol ( value );
}

void bigint_ror_sample ( bigint_element_t *value0, unsigned int size ) {
	bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( void * ) value0 );

	bigint_ror ( value );
}

int bigint_is_zero_sample ( const bigint_element_t *value0,
			    unsigned int size ) {
	const bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( const void * ) value0 );

	return bigint_is_zero ( value );
}

int bigint_is_geq_sample ( const bigint_element_t *value0,
			   const bigint_element_t *reference0,
			   unsigned int size ) {
	const bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( const void * ) value0 );
	const bigint_t ( size ) *reference __attribute__ (( may_alias ))
		= ( ( const void * ) reference0 );

	return bigint_is_geq ( value, reference );
}

int bigint_bit_is_set_sample ( const bigint_element_t *value0,
			       unsigned int size, unsigned int bit ) {
	const bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( const void * ) value0 );

	return bigint_bit_is_set ( value, bit );
}

int bigint_max_set_bit_sample ( const bigint_element_t *value0,
				unsigned int size ) {
	const bigint_t ( size ) *value __attribute__ (( may_alias ))
		= ( ( const void * ) value0 );

	return bigint_max_set_bit ( value );
}

void bigint_grow_sample ( const bigint_element_t *source0,
			  unsigned int source_size, bigint_element_t *dest0,
			  unsigned int dest_size ) {
	const bigint_t ( source_size ) *source __attribute__ (( may_alias ))
		= ( ( const void * ) source0 );
	bigint_t ( dest_size ) *dest __attribute__ (( may_alias ))
		= ( ( void * ) dest0 );

	bigint_grow ( source, dest );
}

void bigint_shrink_sample ( const bigint_element_t *source0,
			    unsigned int source_size, bigint_element_t *dest0,
			    unsigned int dest_size ) {
	const bigint_t ( source_size ) *source __attribute__ (( may_alias ))
		= ( ( const void * ) source0 );
	bigint_t ( dest_size ) *dest __attribute__ (( may_alias ))
		= ( ( void * ) dest0 );

	bigint_shrink ( source, dest );
}

void bigint_multiply_sample ( const bigint_element_t *multiplicand0,
			      const bigint_element_t *multiplier0,
			      bigint_element_t *result0,
			      unsigned int size ) {
	const bigint_t ( size ) *multiplicand __attribute__ (( may_alias ))
		= ( ( const void * ) multiplicand0 );
	const bigint_t ( size ) *multiplier __attribute__ (( may_alias ))
		= ( ( const void * ) multiplier0 );
	bigint_t ( size * 2 ) *result __attribute__ (( may_alias ))
		= ( ( void * ) result0 );

	bigint_multiply ( multiplicand, multiplier, result );
}

void bigint_mod_multiply_sample ( const bigint_element_t *multiplicand0,
				  const bigint_element_t *multiplier0,
				  const bigint_element_t *modulus0,
				  bigint_element_t *result0,
				  unsigned int size,
				  void *tmp ) {
	const bigint_t ( size ) *multiplicand __attribute__ (( may_alias ))
		= ( ( const void * ) multiplicand0 );
	const bigint_t ( size ) *multiplier __attribute__ (( may_alias ))
		= ( ( const void * ) multiplier0 );
	const bigint_t ( size ) *modulus __attribute__ (( may_alias ))
		= ( ( const void * ) modulus0 );
	bigint_t ( size ) *result __attribute__ (( may_alias ))
		= ( ( void * ) result0 );

	bigint_mod_multiply ( multiplicand, multiplier, modulus, result, tmp );
}

void bigint_mod_exp_sample ( const bigint_element_t *base0,
			     const bigint_element_t *modulus0,
			     const bigint_element_t *exponent0,
			     bigint_element_t *result0,
			     unsigned int size, unsigned int exponent_size,
			     void *tmp ) {
	const bigint_t ( size ) *base __attribute__ (( may_alias ))
		= ( ( const void * ) base0 );
	const bigint_t ( size ) *modulus __attribute__ (( may_alias ))
		= ( ( const void * ) modulus0 );
	const bigint_t ( exponent_size ) *exponent __attribute__ (( may_alias ))
		= ( ( const void * ) exponent0 );
	bigint_t ( size ) *result __attribute__ (( may_alias ))
		= ( ( void * ) result0 );

	bigint_mod_exp ( base, modulus, exponent, result, tmp );
}

/**
 * Report result of big integer addition test
 *
 * @v addend		Big integer to add
 * @v value		Big integer to be added to
 * @v expected		Big integer expected result
 */
#define bigint_add_ok( addend, value, expected ) do {			\
	static const uint8_t addend_raw[] = addend;			\
	static const uint8_t value_raw[] = value;			\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) addend_temp;					\
	bigint_t ( size ) value_temp;					\
	{} /* Fix emacs alignment */					\
									\
	assert ( bigint_size ( &addend_temp ) ==			\
		 bigint_size ( &value_temp ) );				\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	bigint_init ( &addend_temp, addend_raw,				\
		      sizeof ( addend_raw ) );				\
	DBG ( "Add:\n" );						\
	DBG_HDA ( 0, &addend_temp, sizeof ( addend_temp ) );		\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_add ( &addend_temp, &value_temp );			\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_done ( &value_temp, result_raw, sizeof ( result_raw ) );	\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Report result of big integer subtraction test
 *
 * @v subtrahend	Big integer to subtract
 * @v value		Big integer to be subtracted from
 * @v expected		Big integer expected result
 */
#define bigint_subtract_ok( subtrahend, value, expected ) do {		\
	static const uint8_t subtrahend_raw[] = subtrahend;		\
	static const uint8_t value_raw[] = value;			\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) subtrahend_temp;				\
	bigint_t ( size ) value_temp;					\
	{} /* Fix emacs alignment */					\
									\
	assert ( bigint_size ( &subtrahend_temp ) ==			\
		 bigint_size ( &value_temp ) );				\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	bigint_init ( &subtrahend_temp, subtrahend_raw,			\
		      sizeof ( subtrahend_raw ) );			\
	DBG ( "Subtract:\n" );						\
	DBG_HDA ( 0, &subtrahend_temp, sizeof ( subtrahend_temp ) );	\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_subtract ( &subtrahend_temp, &value_temp );		\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_done ( &value_temp, result_raw, sizeof ( result_raw ) );	\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Report result of big integer left rotation test
 *
 * @v value		Big integer
 * @v expected		Big integer expected result
 */
#define bigint_rol_ok( value, expected ) do {				\
	static const uint8_t value_raw[] = value;			\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) value_temp;					\
	{} /* Fix emacs alignment */					\
									\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	DBG ( "Rotate left:\n" );					\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_rol ( &value_temp );					\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_done ( &value_temp, result_raw, sizeof ( result_raw ) );	\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Report result of big integer right rotation test
 *
 * @v value		Big integer
 * @v expected		Big integer expected result
 */
#define bigint_ror_ok( value, expected ) do {				\
	static const uint8_t value_raw[] = value;			\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) value_temp;					\
	{} /* Fix emacs alignment */					\
									\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	DBG ( "Rotate right:\n" );					\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_ror ( &value_temp );					\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bigint_done ( &value_temp, result_raw, sizeof ( result_raw ) );	\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Report result of big integer zero comparison test
 *
 * @v value		Big integer
 * @v expected		Expected result
 */
#define bigint_is_zero_ok( value, expected ) do {			\
	static const uint8_t value_raw[] = value;			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) value_temp;					\
	int is_zero;							\
	{} /* Fix emacs alignment */					\
									\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	DBG ( "Zero comparison:\n" );					\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	is_zero = bigint_is_zero ( &value_temp );			\
	DBG ( "...is %szero\n", ( is_zero ? "" : "not " ) );		\
	ok ( ( !! is_zero ) == ( !! (expected) ) );			\
	} while ( 0 )

/**
 * Report result of big integer greater-than-or-equal comparison test
 *
 * @v value		Big integer
 * @v reference		Reference big integer
 * @v expected		Expected result
 */
#define bigint_is_geq_ok( value, reference, expected ) do {		\
	static const uint8_t value_raw[] = value;			\
	static const uint8_t reference_raw[] = reference;		\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) value_temp;					\
	bigint_t ( size ) reference_temp;				\
	int is_geq;							\
	{} /* Fix emacs alignment */					\
									\
	assert ( bigint_size ( &reference_temp ) ==			\
		 bigint_size ( &value_temp ) );				\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	bigint_init ( &reference_temp, reference_raw,			\
		      sizeof ( reference_raw ) );			\
	DBG ( "Greater-than-or-equal comparison:\n" );			\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	DBG_HDA ( 0, &reference_temp, sizeof ( reference_temp ) );	\
	is_geq = bigint_is_geq ( &value_temp, &reference_temp );	\
	DBG ( "...is %sgreater than or equal\n",			\
	      ( is_geq ? "" : "not " ) );				\
	ok ( ( !! is_geq ) == ( !! (expected) ) );			\
	} while ( 0 )

/**
 * Report result of big integer bit-set test
 *
 * @v value		Big integer
 * @v bit		Bit to test
 * @v expected		Expected result
 */
#define bigint_bit_is_set_ok( value, bit, expected ) do {		\
	static const uint8_t value_raw[] = value;			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) value_temp;					\
	int bit_is_set;							\
	{} /* Fix emacs alignment */					\
									\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	DBG ( "Bit set:\n" );						\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	bit_is_set = bigint_bit_is_set ( &value_temp, bit );		\
	DBG ( "...bit %d is %sset\n", bit,				\
	      ( bit_is_set ? "" : "not " ) );				\
	ok ( ( !! bit_is_set ) == ( !! (expected) ) );			\
	} while ( 0 )

/**
 * Report result of big integer maximum set bit test
 *
 * @v value		Big integer
 * @v expected		Expected result
 */
#define bigint_max_set_bit_ok( value, expected ) do {			\
	static const uint8_t value_raw[] = value;			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( value_raw ) );		\
	bigint_t ( size ) value_temp;					\
	int max_set_bit;						\
	{} /* Fix emacs alignment */					\
									\
	bigint_init ( &value_temp, value_raw, sizeof ( value_raw ) );	\
	DBG ( "Maximum set bit:\n" );					\
	DBG_HDA ( 0, &value_temp, sizeof ( value_temp ) );		\
	max_set_bit = bigint_max_set_bit ( &value_temp );		\
	DBG ( "...maximum set bit is bit %d\n", ( max_set_bit - 1 ) );	\
	ok ( max_set_bit == (expected) );				\
	} while ( 0 )

/**
 * Report result of big integer multiplication test
 *
 * @v multiplicand	Big integer to be multiplied
 * @v multiplier	Big integer to be multiplied
 * @v expected		Big integer expected result
 */
#define bigint_multiply_ok( multiplicand, multiplier, expected ) do {	\
	static const uint8_t multiplicand_raw[] = multiplicand;		\
	static const uint8_t multiplier_raw[] = multiplier;		\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( multiplicand_raw ) );	\
	bigint_t ( size ) multiplicand_temp;				\
	bigint_t ( size ) multiplier_temp;				\
	bigint_t ( size * 2 ) result_temp;				\
	{} /* Fix emacs alignment */					\
									\
	assert ( bigint_size ( &multiplier_temp ) ==			\
		 bigint_size ( &multiplicand_temp ) );			\
	assert ( bigint_size ( &result_temp ) ==			\
		 ( 2 * bigint_size ( &multiplicand_temp ) ) );		\
	bigint_init ( &multiplicand_temp, multiplicand_raw,		\
		      sizeof ( multiplicand_raw ) );			\
	bigint_init ( &multiplier_temp, multiplier_raw,			\
		      sizeof ( multiplier_raw ) );			\
	DBG ( "Multiply:\n" );						\
	DBG_HDA ( 0, &multiplicand_temp, sizeof ( multiplicand_temp ) );\
	DBG_HDA ( 0, &multiplier_temp, sizeof ( multiplier_temp ) );	\
	bigint_multiply ( &multiplicand_temp, &multiplier_temp,		\
			  &result_temp );				\
	DBG_HDA ( 0, &result_temp, sizeof ( result_temp ) );		\
	bigint_done ( &result_temp, result_raw, sizeof ( result_raw ) );\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Report result of big integer modular multiplication test
 *
 * @v multiplicand	Big integer to be multiplied
 * @v multiplier	Big integer to be multiplied
 * @v modulus		Big integer modulus
 * @v expected		Big integer expected result
 */
#define bigint_mod_multiply_ok( multiplicand, multiplier, modulus,	\
				expected ) do {				\
	static const uint8_t multiplicand_raw[] = multiplicand;		\
	static const uint8_t multiplier_raw[] = multiplier;		\
	static const uint8_t modulus_raw[] = modulus;			\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( multiplicand_raw ) );	\
	bigint_t ( size ) multiplicand_temp;				\
	bigint_t ( size ) multiplier_temp;				\
	bigint_t ( size ) modulus_temp;					\
	bigint_t ( size ) result_temp;					\
	size_t tmp_len = bigint_mod_multiply_tmp_len ( &modulus_temp ); \
	uint8_t tmp[tmp_len];						\
	{} /* Fix emacs alignment */					\
									\
	assert ( bigint_size ( &multiplier_temp ) ==			\
		 bigint_size ( &multiplicand_temp ) );			\
	assert ( bigint_size ( &multiplier_temp ) ==			\
		 bigint_size ( &modulus_temp ) );			\
	assert ( bigint_size ( &multiplier_temp ) ==			\
		 bigint_size ( &result_temp ) );			\
	bigint_init ( &multiplicand_temp, multiplicand_raw,		\
		      sizeof ( multiplicand_raw ) );			\
	bigint_init ( &multiplier_temp, multiplier_raw,			\
		      sizeof ( multiplier_raw ) );			\
	bigint_init ( &modulus_temp, modulus_raw,			\
		      sizeof ( modulus_raw ) );				\
	DBG ( "Modular multiply:\n" );					\
	DBG_HDA ( 0, &multiplicand_temp, sizeof ( multiplicand_temp ) );\
	DBG_HDA ( 0, &multiplier_temp, sizeof ( multiplier_temp ) );	\
	DBG_HDA ( 0, &modulus_temp, sizeof ( modulus_temp ) );		\
	bigint_mod_multiply ( &multiplicand_temp, &multiplier_temp,	\
			      &modulus_temp, &result_temp, tmp );	\
	DBG_HDA ( 0, &result_temp, sizeof ( result_temp ) );		\
	bigint_done ( &result_temp, result_raw, sizeof ( result_raw ) );\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Report result of big integer modular exponentiation test
 *
 * @v base		Big integer base
 * @v modulus		Big integer modulus
 * @v exponent		Big integer exponent
 * @v expected		Big integer expected result
 */
#define bigint_mod_exp_ok( base, modulus, exponent, expected ) do {	\
	static const uint8_t base_raw[] = base;				\
	static const uint8_t modulus_raw[] = modulus;			\
	static const uint8_t exponent_raw[] = exponent;			\
	static const uint8_t expected_raw[] = expected;			\
	uint8_t result_raw[ sizeof ( expected_raw ) ];			\
	unsigned int size =						\
		bigint_required_size ( sizeof ( base_raw ) );		\
	unsigned int exponent_size =					\
		bigint_required_size ( sizeof ( exponent_raw ) );	\
	bigint_t ( size ) base_temp;					\
	bigint_t ( size ) modulus_temp;					\
	bigint_t ( exponent_size ) exponent_temp;			\
	bigint_t ( size ) result_temp;					\
	size_t tmp_len = bigint_mod_exp_tmp_len ( &modulus_temp,	\
						  &exponent_temp );	\
	uint8_t tmp[tmp_len];						\
	{} /* Fix emacs alignment */					\
									\
	assert ( bigint_size ( &modulus_temp ) ==			\
		 bigint_size ( &base_temp ) );				\
	assert ( bigint_size ( &modulus_temp ) ==			\
		 bigint_size ( &result_temp ) );			\
	bigint_init ( &base_temp, base_raw, sizeof ( base_raw ) );	\
	bigint_init ( &modulus_temp, modulus_raw,			\
		      sizeof ( modulus_raw ) );				\
	bigint_init ( &exponent_temp, exponent_raw,			\
		      sizeof ( exponent_raw ) );			\
	DBG ( "Modular exponentiation:\n" );				\
	DBG_HDA ( 0, &base_temp, sizeof ( base_temp ) );		\
	DBG_HDA ( 0, &modulus_temp, sizeof ( modulus_temp ) );		\
	DBG_HDA ( 0, &exponent_temp, sizeof ( exponent_temp ) );	\
	bigint_mod_exp ( &base_temp, &modulus_temp, &exponent_temp,	\
			 &result_temp, tmp );				\
	DBG_HDA ( 0, &result_temp, sizeof ( result_temp ) );		\
	bigint_done ( &result_temp, result_raw, sizeof ( result_raw ) );\
									\
	ok ( memcmp ( result_raw, expected_raw,				\
		      sizeof ( result_raw ) ) == 0 );			\
	} while ( 0 )

/**
 * Perform big integer self-tests
 *
 */
static void bigint_test_exec ( void ) {

	bigint_add_ok ( BIGINT ( 0x8a ),
			BIGINT ( 0x43 ),
			BIGINT ( 0xcd ) );
	bigint_add_ok ( BIGINT ( 0xc5, 0x7b ),
			BIGINT ( 0xd6, 0xb1 ),
			BIGINT ( 0x9c, 0x2c ) );
	bigint_add_ok ( BIGINT ( 0xf9, 0xd9, 0xdc ),
			BIGINT ( 0x6d, 0x4b, 0xca ),
			BIGINT ( 0x67, 0x25, 0xa6 ) );
	bigint_add_ok ( BIGINT ( 0xdd, 0xc2, 0x20, 0x5f ),
			BIGINT ( 0x80, 0x32, 0xc4, 0xb0 ),
			BIGINT ( 0x5d, 0xf4, 0xe5, 0x0f ) );
	bigint_add_ok ( BIGINT ( 0x01, 0xed, 0x45, 0x4b, 0x41, 0xeb, 0x4c,
				 0x2e, 0x53, 0x07, 0x15, 0x51, 0x56, 0x47,
				 0x29, 0xfc, 0x9c, 0xbd, 0xbd, 0xfb, 0x1b,
				 0xd1, 0x1d ),
			BIGINT ( 0x73, 0xed, 0xfc, 0x35, 0x31, 0x22, 0xd7,
				 0xb1, 0xea, 0x91, 0x5a, 0xe4, 0xba, 0xbc,
				 0xa1, 0x38, 0x72, 0xae, 0x4b, 0x1c, 0xc1,
				 0x05, 0xb3 ),
			BIGINT ( 0x75, 0xdb, 0x41, 0x80, 0x73, 0x0e, 0x23,
				 0xe0, 0x3d, 0x98, 0x70, 0x36, 0x11, 0x03,
				 0xcb, 0x35, 0x0f, 0x6c, 0x09, 0x17, 0xdc,
				 0xd6, 0xd0 ) );
	bigint_add_ok ( BIGINT ( 0x06, 0x8e, 0xd6, 0x18, 0xbb, 0x4b, 0x0c,
				 0xc5, 0x85, 0xde, 0xee, 0x9b, 0x3f, 0x65,
				 0x63, 0x86, 0xf5, 0x5a, 0x9f, 0xa2, 0xd7,
				 0xb2, 0xc7, 0xb6, 0x1d, 0x28, 0x6c, 0x50,
				 0x47, 0x10, 0x0a, 0x0e, 0x86, 0xcd, 0x2a,
				 0x64, 0xdc, 0xe6, 0x9d, 0x96, 0xd8, 0xf4,
				 0x56, 0x46, 0x6f, 0xbb, 0x7b, 0x64, 0x6f,
				 0xdc, 0x2a, 0xd1, 0x3b, 0xcc, 0x03, 0x85,
				 0x95, 0xf4, 0xe9, 0x68, 0x1f, 0x5c, 0xc5,
				 0xbf, 0x97, 0x19, 0x12, 0x88, 0x2e, 0x88,
				 0xb9, 0x34, 0xac, 0x74, 0x83, 0x2d, 0x8f,
				 0xb3, 0x97, 0x53, 0x99, 0xf3, 0xb4, 0x8b,
				 0x2d, 0x98, 0x69, 0x8d, 0x19, 0xf0, 0x40,
				 0x66, 0x3f, 0x60, 0x78, 0x34, 0x7f, 0x9b,
				 0xf7, 0x01, 0x74, 0x55, 0xca, 0x63, 0x25,
				 0x7b, 0x86, 0xe9, 0x73, 0xfd, 0x5d, 0x77,
				 0x32, 0x5e, 0x9e, 0x42, 0x53, 0xb6, 0x35,
				 0x92, 0xb9, 0xd7, 0x1b, 0xf7, 0x16, 0x55,
				 0xf6, 0xe2 ),
			BIGINT ( 0x3f, 0x8f, 0x62, 0x21, 0x4a, 0x7a, 0xa2,
				 0xef, 0xa8, 0x79, 0x9b, 0x73, 0xac, 0xde,
				 0x72, 0xe4, 0xfc, 0x3c, 0xd3, 0xa9, 0x44,
				 0x1a, 0x6a, 0x02, 0x76, 0xe3, 0x78, 0x4d,
				 0x2e, 0x07, 0x9b, 0xb6, 0x3d, 0x5d, 0xc5,
				 0xcd, 0x68, 0x23, 0x4b, 0x5f, 0x89, 0x0e,
				 0xd7, 0xa7, 0xff, 0x18, 0x80, 0xdc, 0xfb,
				 0x34, 0x45, 0xca, 0x4b, 0xdb, 0x8a, 0x19,
				 0xcb, 0xc9, 0xe5, 0xa1, 0x63, 0xa2, 0x0d,
				 0x56, 0xc4, 0xf9, 0x51, 0x1b, 0x88, 0x4e,
				 0x36, 0xab, 0x15, 0x4d, 0x8f, 0xdc, 0x08,
				 0xc4, 0x4d, 0x43, 0xc7, 0x2b, 0xc9, 0x5c,
				 0x05, 0x26, 0xe3, 0x46, 0xf0, 0x64, 0xaa,
				 0x02, 0xa4, 0xbe, 0x3a, 0xd1, 0xca, 0x07,
				 0x6a, 0x6e, 0x62, 0xf4, 0x57, 0x71, 0x96,
				 0xec, 0xf0, 0x0b, 0xac, 0xa4, 0x4a, 0xa3,
				 0x6d, 0x01, 0xba, 0xbd, 0x62, 0xc0, 0x10,
				 0x54, 0x33, 0x8a, 0x71, 0xef, 0xaa, 0x1c,
				 0x25, 0x25 ),
			BIGINT ( 0x46, 0x1e, 0x38, 0x3a, 0x05, 0xc5, 0xaf,
				 0xb5, 0x2e, 0x58, 0x8a, 0x0e, 0xec, 0x43,
				 0xd6, 0x6b, 0xf1, 0x97, 0x73, 0x4c, 0x1b,
				 0xcd, 0x31, 0xb8, 0x94, 0x0b, 0xe4, 0x9d,
				 0x75, 0x17, 0xa5, 0xc4, 0xc4, 0x2a, 0xf0,
				 0x32, 0x45, 0x09, 0xe8, 0xf6, 0x62, 0x03,
				 0x2d, 0xee, 0x6e, 0xd3, 0xfc, 0x41, 0x6b,
				 0x10, 0x70, 0x9b, 0x87, 0xa7, 0x8d, 0x9f,
				 0x61, 0xbe, 0xcf, 0x09, 0x82, 0xfe, 0xd3,
				 0x16, 0x5c, 0x12, 0x63, 0xa3, 0xb6, 0xd6,
				 0xef, 0xdf, 0xc1, 0xc2, 0x13, 0x09, 0x98,
				 0x77, 0xe4, 0x97, 0x61, 0x1f, 0x7d, 0xe7,
				 0x32, 0xbf, 0x4c, 0xd4, 0x0a, 0x54, 0xea,
				 0x68, 0xe4, 0x1e, 0xb3, 0x06, 0x49, 0xa3,
				 0x61, 0x6f, 0xd7, 0x4a, 0x21, 0xd4, 0xbc,
				 0x68, 0x76, 0xf5, 0x20, 0xa1, 0xa8, 0x1a,
				 0x9f, 0x60, 0x58, 0xff, 0xb6, 0x76, 0x45,
				 0xe6, 0xed, 0x61, 0x8d, 0xe6, 0xc0, 0x72,
				 0x1c, 0x07 ) );
	bigint_subtract_ok ( BIGINT ( 0x83 ),
			     BIGINT ( 0x50 ),
			     BIGINT ( 0xcd ) );
	bigint_subtract_ok ( BIGINT ( 0x2c, 0x7c ),
			     BIGINT ( 0x49, 0x0e ),
			     BIGINT ( 0x1c, 0x92 ) );
	bigint_subtract_ok ( BIGINT ( 0x9c, 0x30, 0xbf ),
			     BIGINT ( 0xde, 0x4e, 0x07 ),
			     BIGINT ( 0x42, 0x1d, 0x48 ) );
	bigint_subtract_ok ( BIGINT ( 0xbb, 0x77, 0x32, 0x5a ),
			     BIGINT ( 0x5a, 0xd5, 0xfe, 0x28 ),
			     BIGINT ( 0x9f, 0x5e, 0xcb, 0xce ) );
	bigint_subtract_ok ( BIGINT ( 0x7b, 0xaa, 0x16, 0xcf, 0x15, 0x87,
				      0xe0, 0x4f, 0x2c, 0xa3, 0xec, 0x2f,
				      0x46, 0xfb, 0x83, 0xc6, 0xe0, 0xee,
				      0x57, 0xfa, 0x04, 0xce, 0xa6 ),
			     BIGINT ( 0x46, 0x55, 0xb6, 0x23, 0x63, 0xd0,
				      0x55, 0xdb, 0x8f, 0xcc, 0x55, 0xa8,
				      0x2f, 0x85, 0xc1, 0x9f, 0x2c, 0x13,
				      0x10, 0x9e, 0x76, 0x3c, 0x11 ),
			     BIGINT ( 0xca, 0xab, 0x9f, 0x54, 0x4e, 0x48,
				      0x75, 0x8c, 0x63, 0x28, 0x69, 0x78,
				      0xe8, 0x8a, 0x3d, 0xd8, 0x4b, 0x24,
				      0xb8, 0xa4, 0x71, 0x6d, 0x6b ) );
	bigint_subtract_ok ( BIGINT ( 0x5b, 0x06, 0x77, 0x7b, 0xfd, 0x34,
				      0x5f, 0x0f, 0xd9, 0xbd, 0x8e, 0x5d,
				      0xc8, 0x4a, 0x70, 0x95, 0x1b, 0xb6,
				      0x48, 0xfb, 0x0e, 0x40, 0xce, 0x06,
				      0x66, 0xcc, 0x29, 0xe9, 0x51, 0x59,
				      0x59, 0xc9, 0x65, 0x07, 0x75, 0xb8,
				      0xd4, 0xcb, 0x07, 0x68, 0x14, 0x48,
				      0xc7, 0x1e, 0xfe, 0xb3, 0x4c, 0xf1,
				      0x10, 0xf0, 0xc7, 0x82, 0x38, 0x4c,
				      0xaf, 0x05, 0x6d, 0x91, 0xc5, 0x18,
				      0xfd, 0x1e, 0x26, 0x1b, 0xef, 0x71,
				      0x70, 0x2e, 0x06, 0x70, 0x8e, 0x54,
				      0xfa, 0x2b, 0x4d, 0x96, 0x85, 0x10,
				      0x03, 0x76, 0xe7, 0x17, 0x59, 0x86,
				      0x6c, 0x8b, 0x24, 0x6e, 0xd9, 0x30,
				      0xf3, 0xd2, 0x9b, 0x62, 0xdc, 0x23,
				      0x54, 0x06, 0x51, 0xb1, 0x95, 0x58,
				      0xec, 0x27, 0xf6, 0x19, 0xae, 0xf4,
				      0x31, 0xec, 0x72, 0x53, 0xcd, 0x32,
				      0xed, 0xf4, 0x25, 0x4a, 0x5b, 0x36,
				      0xa2, 0xb4, 0xa0, 0x29, 0x0c, 0x6b,
				      0x3f, 0xc2 ),
			     BIGINT ( 0x7a, 0xd4, 0x25, 0xf1, 0xb5, 0xf5,
				      0x00, 0x96, 0x47, 0x5b, 0x4f, 0x9f,
				      0x1f, 0x61, 0x69, 0xd9, 0x72, 0x47,
				      0xde, 0xbd, 0x87, 0x5d, 0x50, 0x91,
				      0x69, 0xd8, 0x35, 0xe0, 0x43, 0xd8,
				      0xd5, 0x15, 0xf2, 0xcd, 0x01, 0x73,
				      0x0d, 0x34, 0xf0, 0x34, 0x46, 0x76,
				      0xc0, 0x55, 0x7b, 0x27, 0xf5, 0x7b,
				      0x55, 0xe9, 0xd0, 0x29, 0x0b, 0x4b,
				      0x9f, 0x07, 0xbf, 0x2c, 0x3f, 0xef,
				      0x36, 0x34, 0xde, 0x29, 0x1d, 0x5d,
				      0x84, 0x5a, 0x5d, 0xc1, 0x02, 0x4d,
				      0x56, 0xf1, 0x47, 0x39, 0x37, 0xc9,
				      0xb5, 0x5f, 0x73, 0xec, 0x7c, 0x3d,
				      0xbd, 0xc0, 0xfd, 0x38, 0x6c, 0x91,
				      0x88, 0x4a, 0x0f, 0xee, 0xa1, 0x80,
				      0xf5, 0x6a, 0x7c, 0x8c, 0x02, 0xc3,
				      0x5a, 0xb2, 0x15, 0xa6, 0x2f, 0x6b,
				      0x5b, 0x78, 0xb5, 0xf3, 0xbd, 0xd0,
				      0xc8, 0xbc, 0xb1, 0xbb, 0xe1, 0xce,
				      0x22, 0x80, 0x34, 0x5a, 0x2a, 0x27,
				      0x83, 0xdc ),
			     BIGINT ( 0x1f, 0xcd, 0xae, 0x75, 0xb8, 0xc0,
				      0xa1, 0x86, 0x6d, 0x9d, 0xc1, 0x41,
				      0x57, 0x16, 0xf9, 0x44, 0x56, 0x91,
				      0x95, 0xc2, 0x79, 0x1c, 0x82, 0x8b,
				      0x03, 0x0c, 0x0b, 0xf6, 0xf2, 0x7f,
				      0x7b, 0x4c, 0x8d, 0xc5, 0x8b, 0xba,
				      0x38, 0x69, 0xe8, 0xcc, 0x32, 0x2d,
				      0xf9, 0x36, 0x7c, 0x74, 0xa8, 0x8a,
				      0x44, 0xf9, 0x08, 0xa6, 0xd2, 0xfe,
				      0xf0, 0x02, 0x51, 0x9a, 0x7a, 0xd6,
				      0x39, 0x16, 0xb8, 0x0d, 0x2d, 0xec,
				      0x14, 0x2c, 0x57, 0x50, 0x73, 0xf8,
				      0x5c, 0xc5, 0xf9, 0xa2, 0xb2, 0xb9,
				      0xb1, 0xe8, 0x8c, 0xd5, 0x22, 0xb7,
				      0x51, 0x35, 0xd8, 0xc9, 0x93, 0x60,
				      0x94, 0x77, 0x74, 0x8b, 0xc5, 0x5d,
				      0xa1, 0x64, 0x2a, 0xda, 0x6d, 0x6a,
				      0x6e, 0x8a, 0x1f, 0x8c, 0x80, 0x77,
				      0x29, 0x8c, 0x43, 0x9f, 0xf0, 0x9d,
				      0xda, 0xc8, 0x8c, 0x71, 0x86, 0x97,
				      0x7f, 0xcb, 0x94, 0x31, 0x1d, 0xbc,
				      0x44, 0x1a ) );
	bigint_rol_ok ( BIGINT ( 0xe0 ),
			BIGINT ( 0xc0 ) );
	bigint_rol_ok ( BIGINT ( 0x43, 0x1d ),
			BIGINT ( 0x86, 0x3a ) );
	bigint_rol_ok ( BIGINT ( 0xac, 0xed, 0x9b ),
			BIGINT ( 0x59, 0xdb, 0x36 ) );
	bigint_rol_ok ( BIGINT ( 0x2c, 0xe8, 0x3a, 0x22 ),
			BIGINT ( 0x59, 0xd0, 0x74, 0x44 ) );
	bigint_rol_ok ( BIGINT ( 0x4e, 0x88, 0x4a, 0x05, 0x5e, 0x10, 0xee,
				 0x5b, 0xc6, 0x40, 0x0e, 0x03, 0xd7, 0x0d,
				 0x28, 0xa5, 0x55, 0xb2, 0x50, 0xef, 0x69,
				 0xd1, 0x1d ),
			BIGINT ( 0x9d, 0x10, 0x94, 0x0a, 0xbc, 0x21, 0xdc,
				 0xb7, 0x8c, 0x80, 0x1c, 0x07, 0xae, 0x1a,
				 0x51, 0x4a, 0xab, 0x64, 0xa1, 0xde, 0xd3,
				 0xa2, 0x3a ) );
	bigint_rol_ok ( BIGINT ( 0x07, 0x62, 0x78, 0x70, 0x2e, 0xd4, 0x41,
				 0xaa, 0x9b, 0x50, 0xb1, 0x9a, 0x71, 0xf5,
				 0x1c, 0x2f, 0xe7, 0x0d, 0xf1, 0x46, 0x57,
				 0x04, 0x99, 0x78, 0x4e, 0x84, 0x78, 0xba,
				 0x57, 0xea, 0xa5, 0x43, 0xf7, 0x02, 0xf0,
				 0x7a, 0x22, 0x60, 0x65, 0x42, 0xf2, 0x33,
				 0x7d, 0xe3, 0xa8, 0x1b, 0xc4, 0x14, 0xdb,
				 0xee, 0x4a, 0xf1, 0xe1, 0x52, 0xd4, 0xda,
				 0x23, 0xed, 0x13, 0x5d, 0xea, 0xcf, 0xf6,
				 0x5e, 0x39, 0x84, 0xe2, 0xb3, 0xa2, 0x05,
				 0xba, 0xd9, 0x49, 0x8e, 0x75, 0x1d, 0xdb,
				 0xe6, 0xb1, 0x6e, 0xda, 0x0a, 0x83, 0xd0,
				 0x6e, 0xcf, 0x7a, 0x66, 0xb7, 0x64, 0x84,
				 0xf5, 0x09, 0x5a, 0xa8, 0x11, 0x93, 0xf3,
				 0x4f, 0x02, 0x28, 0x00, 0x3a, 0xf0, 0xa9,
				 0x08, 0x77, 0x04, 0xf5, 0x04, 0xcd, 0x6b,
				 0x24, 0xbe, 0x0f, 0x6d, 0xe3, 0xb2, 0xd3,
				 0x07, 0x68, 0xe9, 0x00, 0x59, 0xa0, 0xe4,
				 0x9e, 0x5e ),
			BIGINT ( 0x0e, 0xc4, 0xf0, 0xe0, 0x5d, 0xa8, 0x83,
				 0x55, 0x36, 0xa1, 0x63, 0x34, 0xe3, 0xea,
				 0x38, 0x5f, 0xce, 0x1b, 0xe2, 0x8c, 0xae,
				 0x09, 0x32, 0xf0, 0x9d, 0x08, 0xf1, 0x74,
				 0xaf, 0xd5, 0x4a, 0x87, 0xee, 0x05, 0xe0,
				 0xf4, 0x44, 0xc0, 0xca, 0x85, 0xe4, 0x66,
				 0xfb, 0xc7, 0x50, 0x37, 0x88, 0x29, 0xb7,
				 0xdc, 0x95, 0xe3, 0xc2, 0xa5, 0xa9, 0xb4,
				 0x47, 0xda, 0x26, 0xbb, 0xd5, 0x9f, 0xec,
				 0xbc, 0x73, 0x09, 0xc5, 0x67, 0x44, 0x0b,
				 0x75, 0xb2, 0x93, 0x1c, 0xea, 0x3b, 0xb7,
				 0xcd, 0x62, 0xdd, 0xb4, 0x15, 0x07, 0xa0,
				 0xdd, 0x9e, 0xf4, 0xcd, 0x6e, 0xc9, 0x09,
				 0xea, 0x12, 0xb5, 0x50, 0x23, 0x27, 0xe6,
				 0x9e, 0x04, 0x50, 0x00, 0x75, 0xe1, 0x52,
				 0x10, 0xee, 0x09, 0xea, 0x09, 0x9a, 0xd6,
				 0x49, 0x7c, 0x1e, 0xdb, 0xc7, 0x65, 0xa6,
				 0x0e, 0xd1, 0xd2, 0x00, 0xb3, 0x41, 0xc9,
				 0x3c, 0xbc ) );
	bigint_ror_ok ( BIGINT ( 0x8f ),
			BIGINT ( 0x47 ) );
	bigint_ror_ok ( BIGINT ( 0xaa, 0x1d ),
			BIGINT ( 0x55, 0x0e ) );
	bigint_ror_ok ( BIGINT ( 0xf0, 0xbd, 0x68 ),
			BIGINT ( 0x78, 0x5e, 0xb4 ) );
	bigint_ror_ok ( BIGINT ( 0x33, 0xa0, 0x3d, 0x95 ),
			BIGINT ( 0x19, 0xd0, 0x1e, 0xca ) );
	bigint_ror_ok ( BIGINT ( 0xa1, 0xf4, 0xb9, 0x64, 0x91, 0x99, 0xa1,
				 0xf4, 0xae, 0xeb, 0x71, 0x97, 0x1b, 0x71,
				 0x09, 0x38, 0x3f, 0x8f, 0xc5, 0x3a, 0xb9,
				 0x75, 0x94 ),
			BIGINT ( 0x50, 0xfa, 0x5c, 0xb2, 0x48, 0xcc, 0xd0,
				 0xfa, 0x57, 0x75, 0xb8, 0xcb, 0x8d, 0xb8,
				 0x84, 0x9c, 0x1f, 0xc7, 0xe2, 0x9d, 0x5c,
				 0xba, 0xca ) );
	bigint_ror_ok ( BIGINT ( 0xc0, 0xb3, 0x78, 0x46, 0x69, 0x6e, 0x35,
				 0x94, 0xed, 0x28, 0xdc, 0xfd, 0xf6, 0xdb,
				 0x2d, 0x24, 0xcb, 0xa4, 0x6f, 0x0e, 0x58,
				 0x89, 0x04, 0xec, 0xc8, 0x0c, 0x2d, 0xb3,
				 0x58, 0xa7, 0x22, 0x6d, 0x93, 0xe0, 0xb8,
				 0x48, 0x6a, 0x3f, 0x04, 0x7e, 0xbe, 0xb8,
				 0xa7, 0x84, 0xf5, 0xc9, 0x2f, 0x60, 0x9e,
				 0x7c, 0xbc, 0xaf, 0x28, 0x89, 0x2f, 0xaa,
				 0xd1, 0x82, 0x77, 0xa4, 0xdf, 0xf3, 0x4a,
				 0xc6, 0xed, 0xa3, 0x07, 0xb4, 0xa9, 0xfd,
				 0xef, 0xf8, 0x20, 0xb9, 0xb3, 0xff, 0x35,
				 0x27, 0xed, 0x02, 0xea, 0xec, 0x63, 0xc0,
				 0x46, 0x97, 0xc0, 0x4c, 0xca, 0x89, 0xca,
				 0x14, 0xe8, 0xe0, 0x02, 0x14, 0x44, 0x46,
				 0xf3, 0x2f, 0xbc, 0x6a, 0x28, 0xa2, 0xbe,
				 0x20, 0xc8, 0xaa, 0x0f, 0xd9, 0x51, 0x8e,
				 0x8d, 0x51, 0x29, 0x61, 0xef, 0x48, 0xae,
				 0x3e, 0xe5, 0x10, 0xbf, 0xda, 0x9b, 0x92,
				 0xb3, 0x77 ),
			BIGINT ( 0x60, 0x59, 0xbc, 0x23, 0x34, 0xb7, 0x1a,
				 0xca, 0x76, 0x94, 0x6e, 0x7e, 0xfb, 0x6d,
				 0x96, 0x92, 0x65, 0xd2, 0x37, 0x87, 0x2c,
				 0x44, 0x82, 0x76, 0x64, 0x06, 0x16, 0xd9,
				 0xac, 0x53, 0x91, 0x36, 0xc9, 0xf0, 0x5c,
				 0x24, 0x35, 0x1f, 0x82, 0x3f, 0x5f, 0x5c,
				 0x53, 0xc2, 0x7a, 0xe4, 0x97, 0xb0, 0x4f,
				 0x3e, 0x5e, 0x57, 0x94, 0x44, 0x97, 0xd5,
				 0x68, 0xc1, 0x3b, 0xd2, 0x6f, 0xf9, 0xa5,
				 0x63, 0x76, 0xd1, 0x83, 0xda, 0x54, 0xfe,
				 0xf7, 0xfc, 0x10, 0x5c, 0xd9, 0xff, 0x9a,
				 0x93, 0xf6, 0x81, 0x75, 0x76, 0x31, 0xe0,
				 0x23, 0x4b, 0xe0, 0x26, 0x65, 0x44, 0xe5,
				 0x0a, 0x74, 0x70, 0x01, 0x0a, 0x22, 0x23,
				 0x79, 0x97, 0xde, 0x35, 0x14, 0x51, 0x5f,
				 0x10, 0x64, 0x55, 0x07, 0xec, 0xa8, 0xc7,
				 0x46, 0xa8, 0x94, 0xb0, 0xf7, 0xa4, 0x57,
				 0x1f, 0x72, 0x88, 0x5f, 0xed, 0x4d, 0xc9,
				 0x59, 0xbb ) );
	bigint_is_zero_ok ( BIGINT ( 0x9b ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0x5a, 0x9d ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0x5f, 0x80, 0x78 ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0xa0, 0x52, 0x47, 0x2e ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0x18, 0x08, 0x49, 0xdb, 0x7b, 0x5c,
				     0xe7, 0x41, 0x07, 0xdf, 0xed, 0xf9,
				     0xd3, 0x92, 0x0d, 0x75, 0xa6, 0xb0,
				     0x14, 0xfa, 0xdd, 0xfd, 0x82 ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0x04, 0x04, 0xb5, 0xf5, 0x01, 0xae,
				     0x2b, 0x91, 0xa7, 0xc1, 0x49, 0x97,
				     0x3f, 0x45, 0x53, 0x52, 0xb8, 0x52,
				     0xf1, 0x62, 0xa5, 0x21, 0x18, 0xd4,
				     0xb0, 0xb4, 0x8a, 0x17, 0x0e, 0xe8,
				     0xeb, 0xaa, 0x28, 0xae, 0x3d, 0x8e,
				     0xe3, 0x6c, 0xd0, 0x01, 0x0c, 0x54,
				     0xca, 0x23, 0xbb, 0x06, 0xcd, 0x7a,
				     0x61, 0x89, 0x38, 0x34, 0x6e, 0xc7,
				     0xc2, 0xee, 0xb1, 0x80, 0x61, 0x0e,
				     0xc6, 0x8d, 0x65, 0xa0, 0xeb, 0x34,
				     0xe9, 0x63, 0x09, 0x4c, 0x20, 0xac,
				     0x42, 0xe3, 0x35, 0xa2, 0x3e, 0x3b,
				     0x2e, 0x18, 0x70, 0x45, 0x7c, 0xab,
				     0x42, 0xcc, 0xe0, 0x9e, 0x7c, 0x42,
				     0xd1, 0xda, 0x6c, 0x51, 0x10, 0x1e,
				     0x0e, 0x3f, 0xe5, 0xd6, 0xd8, 0x56,
				     0x08, 0xb2, 0x3b, 0x15, 0xc4, 0x7c,
				     0x0c, 0x7e, 0xaf, 0x7b, 0x9d, 0xd6,
				     0x2b, 0xc0, 0x2f, 0xa2, 0xa3, 0xa3,
				     0x77, 0x58, 0x1b, 0xe9, 0xa8, 0x9a,
				     0x23, 0x7f ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0x00 ),
			    1 );
	bigint_is_zero_ok ( BIGINT ( 0x00, 0x00 ),
			    1 );
	bigint_is_zero_ok ( BIGINT ( 0x00, 0x00, 0x00 ),
			    1 );
	bigint_is_zero_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00 ),
			    1 );
	bigint_is_zero_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00 ),
			    1 );
	bigint_is_zero_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				     0x00, 0x00 ),
			    1 );
	bigint_is_zero_ok ( BIGINT ( 0xff ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0xff, 0xff ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0xff, 0xff, 0xff ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff ),
			    0 );
	bigint_is_zero_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				     0xff, 0xff ),
			    0 );
	bigint_is_geq_ok ( BIGINT ( 0xa2 ),
			   BIGINT ( 0x58 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0x58 ),
			   BIGINT ( 0xa2 ),
			   0 );
	bigint_is_geq_ok ( BIGINT ( 0xa2 ),
			   BIGINT ( 0xa2 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0x61, 0x29 ),
			   BIGINT ( 0x87, 0xac ),
			   0 );
	bigint_is_geq_ok ( BIGINT ( 0x87, 0xac ),
			   BIGINT ( 0x61, 0x29 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0x61, 0x29 ),
			   BIGINT ( 0x61, 0x29 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0xe6, 0x63, 0x14 ),
			   BIGINT ( 0xb7, 0x2b, 0x76 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0xb7, 0x2b, 0x76 ),
			   BIGINT ( 0xe6, 0x63, 0x14 ),
			   0 );
	bigint_is_geq_ok ( BIGINT ( 0xe6, 0x63, 0x14 ),
			   BIGINT ( 0xe6, 0x63, 0x14 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0xe7, 0x4f, 0xd4, 0x80 ),
			   BIGINT ( 0xb5, 0xf9, 0x9b, 0x90 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0xb5, 0xf9, 0x9b, 0x90 ),
			   BIGINT ( 0xe7, 0x4f, 0xd4, 0x80 ),
			   0 );
	bigint_is_geq_ok ( BIGINT ( 0xe7, 0x4f, 0xd4, 0x80 ),
			   BIGINT ( 0xe7, 0x4f, 0xd4, 0x80 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0xe6, 0x2c, 0x7c, 0x24, 0x78, 0x8f, 0x12,
				    0x20, 0xde, 0xd3, 0x6b, 0xc9, 0x97, 0x2d,
				    0x66, 0x74, 0xe5, 0xb6, 0xf7, 0x8f, 0x2b,
				    0x60, 0x98 ),
			   BIGINT ( 0x77, 0xbc, 0x3b, 0x1b, 0x57, 0x43, 0x3b,
				    0x8c, 0x82, 0xda, 0xb5, 0xc7, 0x18, 0x09,
				    0xb3, 0x59, 0x0e, 0x53, 0x2a, 0xb9, 0xd8,
				    0xa2, 0xb4 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0x77, 0xbc, 0x3b, 0x1b, 0x57, 0x43, 0x3b,
				    0x8c, 0x82, 0xda, 0xb5, 0xc7, 0x18, 0x09,
				    0xb3, 0x59, 0x0e, 0x53, 0x2a, 0xb9, 0xd8,
				    0xa2, 0xb4 ),
			   BIGINT ( 0xe6, 0x2c, 0x7c, 0x24, 0x78, 0x8f, 0x12,
				    0x20, 0xde, 0xd3, 0x6b, 0xc9, 0x97, 0x2d,
				    0x66, 0x74, 0xe5, 0xb6, 0xf7, 0x8f, 0x2b,
				    0x60, 0x98 ),
			   0 );
	bigint_is_geq_ok ( BIGINT ( 0xe6, 0x2c, 0x7c, 0x24, 0x78, 0x8f, 0x12,
				    0x20, 0xde, 0xd3, 0x6b, 0xc9, 0x97, 0x2d,
				    0x66, 0x74, 0xe5, 0xb6, 0xf7, 0x8f, 0x2b,
				    0x60, 0x98 ),
			   BIGINT ( 0xe6, 0x2c, 0x7c, 0x24, 0x78, 0x8f, 0x12,
				    0x20, 0xde, 0xd3, 0x6b, 0xc9, 0x97, 0x2d,
				    0x66, 0x74, 0xe5, 0xb6, 0xf7, 0x8f, 0x2b,
				    0x60, 0x98 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0x2a, 0x98, 0xfd, 0x87, 0x5d, 0x9f, 0xb4,
				    0x8b, 0x5c, 0xcd, 0x5f, 0xcd, 0x53, 0xb3,
				    0xd1, 0x81, 0x6a, 0x9c, 0x93, 0x66, 0x40,
				    0xa7, 0x64, 0xe0, 0x8c, 0xec, 0x96, 0x63,
				    0x4d, 0x29, 0xfa, 0xb1, 0x5d, 0x93, 0x2f,
				    0xf9, 0x36, 0xea, 0x3b, 0xc1, 0xaf, 0x85,
				    0xcb, 0xde, 0x2d, 0xc8, 0x48, 0x33, 0xce,
				    0x7b, 0xa4, 0xa4, 0xda, 0x0f, 0xaa, 0x1b,
				    0xcb, 0xed, 0xbe, 0x3a, 0xa5, 0xbb, 0x73,
				    0x28, 0x04, 0xc6, 0x2a, 0xfb, 0x3a, 0xc3,
				    0xae, 0x42, 0x1f, 0x53, 0x6c, 0xb2, 0x76,
				    0xb7, 0xe2, 0x88, 0xcb, 0x88, 0xcf, 0xf0,
				    0x52, 0x81, 0xd3, 0xb2, 0x1f, 0x56, 0xe1,
				    0xe1, 0x47, 0x93, 0x6f, 0x2b, 0x49, 0xaa,
				    0x50, 0x99, 0x7a, 0xc4, 0x56, 0xb7, 0x13,
				    0x80, 0xf4, 0x73, 0x88, 0xc7, 0x39, 0x83,
				    0x67, 0xc7, 0xcc, 0xb2, 0x28, 0x7a, 0xd3,
				    0xdc, 0x48, 0xea, 0x62, 0x0d, 0xf5, 0x5a,
				    0x27, 0x96 ),
			   BIGINT ( 0xd4, 0x6b, 0x0a, 0x2e, 0x9f, 0xde, 0x4b,
				    0x64, 0xfa, 0x6b, 0x37, 0x73, 0x66, 0x06,
				    0xee, 0x04, 0xef, 0xe6, 0x3c, 0x7d, 0x57,
				    0x22, 0x7f, 0x1f, 0x62, 0x1c, 0x7e, 0x20,
				    0xda, 0x97, 0xd0, 0x27, 0x23, 0xf6, 0x77,
				    0x5b, 0x49, 0x97, 0xe1, 0x65, 0x91, 0x13,
				    0x93, 0xd6, 0x12, 0xc3, 0x66, 0x91, 0x76,
				    0xe8, 0x47, 0x4c, 0x6a, 0x1b, 0xa2, 0x02,
				    0xf8, 0x94, 0xaa, 0xe0, 0x1b, 0x0b, 0x17,
				    0x86, 0x5e, 0xf5, 0x17, 0x23, 0xf5, 0x17,
				    0x91, 0x6b, 0xd7, 0x2f, 0x5a, 0xfe, 0x8a,
				    0x63, 0x28, 0x31, 0x1e, 0x09, 0x60, 0x29,
				    0x5d, 0x55, 0xd8, 0x79, 0xeb, 0x78, 0x36,
				    0x44, 0x69, 0xa4, 0x76, 0xa5, 0x35, 0x30,
				    0xca, 0xc9, 0xf9, 0x62, 0xd7, 0x82, 0x13,
				    0x56, 0xd0, 0x58, 0xfe, 0x16, 0x4b, 0xfb,
				    0xa8, 0x4c, 0xb3, 0xd7, 0xcf, 0x5f, 0x93,
				    0x9d, 0xc4, 0x11, 0xb4, 0xdd, 0xf8, 0x8f,
				    0xe1, 0x11 ),
			   0 );
	bigint_is_geq_ok ( BIGINT ( 0xd4, 0x6b, 0x0a, 0x2e, 0x9f, 0xde, 0x4b,
				    0x64, 0xfa, 0x6b, 0x37, 0x73, 0x66, 0x06,
				    0xee, 0x04, 0xef, 0xe6, 0x3c, 0x7d, 0x57,
				    0x22, 0x7f, 0x1f, 0x62, 0x1c, 0x7e, 0x20,
				    0xda, 0x97, 0xd0, 0x27, 0x23, 0xf6, 0x77,
				    0x5b, 0x49, 0x97, 0xe1, 0x65, 0x91, 0x13,
				    0x93, 0xd6, 0x12, 0xc3, 0x66, 0x91, 0x76,
				    0xe8, 0x47, 0x4c, 0x6a, 0x1b, 0xa2, 0x02,
				    0xf8, 0x94, 0xaa, 0xe0, 0x1b, 0x0b, 0x17,
				    0x86, 0x5e, 0xf5, 0x17, 0x23, 0xf5, 0x17,
				    0x91, 0x6b, 0xd7, 0x2f, 0x5a, 0xfe, 0x8a,
				    0x63, 0x28, 0x31, 0x1e, 0x09, 0x60, 0x29,
				    0x5d, 0x55, 0xd8, 0x79, 0xeb, 0x78, 0x36,
				    0x44, 0x69, 0xa4, 0x76, 0xa5, 0x35, 0x30,
				    0xca, 0xc9, 0xf9, 0x62, 0xd7, 0x82, 0x13,
				    0x56, 0xd0, 0x58, 0xfe, 0x16, 0x4b, 0xfb,
				    0xa8, 0x4c, 0xb3, 0xd7, 0xcf, 0x5f, 0x93,
				    0x9d, 0xc4, 0x11, 0xb4, 0xdd, 0xf8, 0x8f,
				    0xe1, 0x11 ),
			   BIGINT ( 0x2a, 0x98, 0xfd, 0x87, 0x5d, 0x9f, 0xb4,
				    0x8b, 0x5c, 0xcd, 0x5f, 0xcd, 0x53, 0xb3,
				    0xd1, 0x81, 0x6a, 0x9c, 0x93, 0x66, 0x40,
				    0xa7, 0x64, 0xe0, 0x8c, 0xec, 0x96, 0x63,
				    0x4d, 0x29, 0xfa, 0xb1, 0x5d, 0x93, 0x2f,
				    0xf9, 0x36, 0xea, 0x3b, 0xc1, 0xaf, 0x85,
				    0xcb, 0xde, 0x2d, 0xc8, 0x48, 0x33, 0xce,
				    0x7b, 0xa4, 0xa4, 0xda, 0x0f, 0xaa, 0x1b,
				    0xcb, 0xed, 0xbe, 0x3a, 0xa5, 0xbb, 0x73,
				    0x28, 0x04, 0xc6, 0x2a, 0xfb, 0x3a, 0xc3,
				    0xae, 0x42, 0x1f, 0x53, 0x6c, 0xb2, 0x76,
				    0xb7, 0xe2, 0x88, 0xcb, 0x88, 0xcf, 0xf0,
				    0x52, 0x81, 0xd3, 0xb2, 0x1f, 0x56, 0xe1,
				    0xe1, 0x47, 0x93, 0x6f, 0x2b, 0x49, 0xaa,
				    0x50, 0x99, 0x7a, 0xc4, 0x56, 0xb7, 0x13,
				    0x80, 0xf4, 0x73, 0x88, 0xc7, 0x39, 0x83,
				    0x67, 0xc7, 0xcc, 0xb2, 0x28, 0x7a, 0xd3,
				    0xdc, 0x48, 0xea, 0x62, 0x0d, 0xf5, 0x5a,
				    0x27, 0x96 ),
			   1 );
	bigint_is_geq_ok ( BIGINT ( 0x2a, 0x98, 0xfd, 0x87, 0x5d, 0x9f, 0xb4,
				    0x8b, 0x5c, 0xcd, 0x5f, 0xcd, 0x53, 0xb3,
				    0xd1, 0x81, 0x6a, 0x9c, 0x93, 0x66, 0x40,
				    0xa7, 0x64, 0xe0, 0x8c, 0xec, 0x96, 0x63,
				    0x4d, 0x29, 0xfa, 0xb1, 0x5d, 0x93, 0x2f,
				    0xf9, 0x36, 0xea, 0x3b, 0xc1, 0xaf, 0x85,
				    0xcb, 0xde, 0x2d, 0xc8, 0x48, 0x33, 0xce,
				    0x7b, 0xa4, 0xa4, 0xda, 0x0f, 0xaa, 0x1b,
				    0xcb, 0xed, 0xbe, 0x3a, 0xa5, 0xbb, 0x73,
				    0x28, 0x04, 0xc6, 0x2a, 0xfb, 0x3a, 0xc3,
				    0xae, 0x42, 0x1f, 0x53, 0x6c, 0xb2, 0x76,
				    0xb7, 0xe2, 0x88, 0xcb, 0x88, 0xcf, 0xf0,
				    0x52, 0x81, 0xd3, 0xb2, 0x1f, 0x56, 0xe1,
				    0xe1, 0x47, 0x93, 0x6f, 0x2b, 0x49, 0xaa,
				    0x50, 0x99, 0x7a, 0xc4, 0x56, 0xb7, 0x13,
				    0x80, 0xf4, 0x73, 0x88, 0xc7, 0x39, 0x83,
				    0x67, 0xc7, 0xcc, 0xb2, 0x28, 0x7a, 0xd3,
				    0xdc, 0x48, 0xea, 0x62, 0x0d, 0xf5, 0x5a,
				    0x27, 0x96 ),
			   BIGINT ( 0x2a, 0x98, 0xfd, 0x87, 0x5d, 0x9f, 0xb4,
				    0x8b, 0x5c, 0xcd, 0x5f, 0xcd, 0x53, 0xb3,
				    0xd1, 0x81, 0x6a, 0x9c, 0x93, 0x66, 0x40,
				    0xa7, 0x64, 0xe0, 0x8c, 0xec, 0x96, 0x63,
				    0x4d, 0x29, 0xfa, 0xb1, 0x5d, 0x93, 0x2f,
				    0xf9, 0x36, 0xea, 0x3b, 0xc1, 0xaf, 0x85,
				    0xcb, 0xde, 0x2d, 0xc8, 0x48, 0x33, 0xce,
				    0x7b, 0xa4, 0xa4, 0xda, 0x0f, 0xaa, 0x1b,
				    0xcb, 0xed, 0xbe, 0x3a, 0xa5, 0xbb, 0x73,
				    0x28, 0x04, 0xc6, 0x2a, 0xfb, 0x3a, 0xc3,
				    0xae, 0x42, 0x1f, 0x53, 0x6c, 0xb2, 0x76,
				    0xb7, 0xe2, 0x88, 0xcb, 0x88, 0xcf, 0xf0,
				    0x52, 0x81, 0xd3, 0xb2, 0x1f, 0x56, 0xe1,
				    0xe1, 0x47, 0x93, 0x6f, 0x2b, 0x49, 0xaa,
				    0x50, 0x99, 0x7a, 0xc4, 0x56, 0xb7, 0x13,
				    0x80, 0xf4, 0x73, 0x88, 0xc7, 0x39, 0x83,
				    0x67, 0xc7, 0xcc, 0xb2, 0x28, 0x7a, 0xd3,
				    0xdc, 0x48, 0xea, 0x62, 0x0d, 0xf5, 0x5a,
				    0x27, 0x96 ),
			   1 );
	bigint_bit_is_set_ok ( BIGINT ( 0x37 ),
			       0, 1 );
	bigint_bit_is_set_ok ( BIGINT ( 0xe6, 0xcb ),
			       0, 1 );
	bigint_bit_is_set_ok ( BIGINT ( 0xd9, 0x0c, 0x5b ),
			       0, 1 );
	bigint_bit_is_set_ok ( BIGINT ( 0x8b, 0x56, 0x89, 0xaf ),
			       0, 1 );
	bigint_bit_is_set_ok ( BIGINT ( 0x25, 0xfc, 0xaf, 0xeb, 0x81, 0xc3,
					0xb8, 0x2f, 0xbb, 0xe3, 0x07, 0xb2,
					0xe2, 0x2a, 0xe2, 0x2d, 0xb4, 0x4d,
					0x6d, 0xec, 0x51, 0xa0, 0x2f ),
			       0, 1 );
	bigint_bit_is_set_ok ( BIGINT ( 0x25, 0xfc, 0xaf, 0xeb, 0x81, 0xc3,
					0xb8, 0x2f, 0xbb, 0xe3, 0x07, 0xb2,
					0xe2, 0x2a, 0xe2, 0x2d, 0xb4, 0x4d,
					0x6d, 0xec, 0x51, 0xa0, 0x2f ),
			       45, 0 );
	bigint_bit_is_set_ok ( BIGINT ( 0x88, 0x04, 0xec, 0xe6, 0xfb, 0x31,
					0x87, 0x43, 0xb2, 0x04, 0x9e, 0x09,
					0xba, 0x3e, 0x6d, 0x64, 0x1a, 0x85,
					0xb6, 0x46, 0x7d, 0x71, 0x3c, 0x06,
					0xd6, 0x40, 0x52, 0x39, 0x95, 0xa1,
					0x06, 0xff, 0x6a, 0x5c, 0xa3, 0x6d,
					0x4a, 0xc9, 0x77, 0x87, 0x75, 0x25,
					0x57, 0x65, 0x72, 0x73, 0x64, 0x7e,
					0xe9, 0x16, 0x17, 0xf3, 0x65, 0x3f,
					0xd5, 0xcc, 0xd7, 0xa2, 0xee, 0xe7,
					0x8d, 0x48, 0xd5, 0x7e, 0xdd, 0x59,
					0x4b, 0xf0, 0x96, 0x8b, 0x21, 0x65,
					0x04, 0x66, 0xc5, 0xff, 0x3e, 0x60,
					0xba, 0x28, 0x38, 0x7d, 0x9c, 0x09,
					0xd1, 0x8e, 0xac, 0x73, 0x8e, 0xf2,
					0x1e, 0xdf, 0x83, 0x6e, 0x54, 0xd5,
					0x34, 0xc1, 0xc6, 0xf9, 0x62, 0x2a,
					0x7d, 0xec, 0x47, 0xf2, 0xfc, 0xa2,
					0x10, 0x0a, 0x67, 0x1b, 0xc6, 0x11,
					0x9d, 0x68, 0x25, 0x8b, 0xb5, 0x9b,
					0x83, 0xf8, 0xa2, 0x11, 0xf5, 0xd4,
					0xcb, 0xe0 ),
			       0, 0 );
	bigint_bit_is_set_ok ( BIGINT ( 0x88, 0x04, 0xec, 0xe6, 0xfb, 0x31,
					0x87, 0x43, 0xb2, 0x04, 0x9e, 0x09,
					0xba, 0x3e, 0x6d, 0x64, 0x1a, 0x85,
					0xb6, 0x46, 0x7d, 0x71, 0x3c, 0x06,
					0xd6, 0x40, 0x52, 0x39, 0x95, 0xa1,
					0x06, 0xff, 0x6a, 0x5c, 0xa3, 0x6d,
					0x4a, 0xc9, 0x77, 0x87, 0x75, 0x25,
					0x57, 0x65, 0x72, 0x73, 0x64, 0x7e,
					0xe9, 0x16, 0x17, 0xf3, 0x65, 0x3f,
					0xd5, 0xcc, 0xd7, 0xa2, 0xee, 0xe7,
					0x8d, 0x48, 0xd5, 0x7e, 0xdd, 0x59,
					0x4b, 0xf0, 0x96, 0x8b, 0x21, 0x65,
					0x04, 0x66, 0xc5, 0xff, 0x3e, 0x60,
					0xba, 0x28, 0x38, 0x7d, 0x9c, 0x09,
					0xd1, 0x8e, 0xac, 0x73, 0x8e, 0xf2,
					0x1e, 0xdf, 0x83, 0x6e, 0x54, 0xd5,
					0x34, 0xc1, 0xc6, 0xf9, 0x62, 0x2a,
					0x7d, 0xec, 0x47, 0xf2, 0xfc, 0xa2,
					0x10, 0x0a, 0x67, 0x1b, 0xc6, 0x11,
					0x9d, 0x68, 0x25, 0x8b, 0xb5, 0x9b,
					0x83, 0xf8, 0xa2, 0x11, 0xf5, 0xd4,
					0xcb, 0xe0 ),
			       45, 1 );
	bigint_bit_is_set_ok ( BIGINT ( 0x88, 0x04, 0xec, 0xe6, 0xfb, 0x31,
					0x87, 0x43, 0xb2, 0x04, 0x9e, 0x09,
					0xba, 0x3e, 0x6d, 0x64, 0x1a, 0x85,
					0xb6, 0x46, 0x7d, 0x71, 0x3c, 0x06,
					0xd6, 0x40, 0x52, 0x39, 0x95, 0xa1,
					0x06, 0xff, 0x6a, 0x5c, 0xa3, 0x6d,
					0x4a, 0xc9, 0x77, 0x87, 0x75, 0x25,
					0x57, 0x65, 0x72, 0x73, 0x64, 0x7e,
					0xe9, 0x16, 0x17, 0xf3, 0x65, 0x3f,
					0xd5, 0xcc, 0xd7, 0xa2, 0xee, 0xe7,
					0x8d, 0x48, 0xd5, 0x7e, 0xdd, 0x59,
					0x4b, 0xf0, 0x96, 0x8b, 0x21, 0x65,
					0x04, 0x66, 0xc5, 0xff, 0x3e, 0x60,
					0xba, 0x28, 0x38, 0x7d, 0x9c, 0x09,
					0xd1, 0x8e, 0xac, 0x73, 0x8e, 0xf2,
					0x1e, 0xdf, 0x83, 0x6e, 0x54, 0xd5,
					0x34, 0xc1, 0xc6, 0xf9, 0x62, 0x2a,
					0x7d, 0xec, 0x47, 0xf2, 0xfc, 0xa2,
					0x10, 0x0a, 0x67, 0x1b, 0xc6, 0x11,
					0x9d, 0x68, 0x25, 0x8b, 0xb5, 0x9b,
					0x83, 0xf8, 0xa2, 0x11, 0xf5, 0xd4,
					0xcb, 0xe0 ),
			       1013, 0 );
	bigint_max_set_bit_ok ( BIGINT ( 0x3a ),
				6 );
	bigint_max_set_bit_ok ( BIGINT ( 0x03 ),
				2 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0xff ),
				8 );
	bigint_max_set_bit_ok ( BIGINT ( 0x20, 0x30 ),
				14 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x10 ),
				5 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0xff, 0xff ),
				16 );
	bigint_max_set_bit_ok ( BIGINT ( 0x06, 0xdb, 0x7a ),
				19 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0xff, 0xff, 0xff ),
				24 );
	bigint_max_set_bit_ok ( BIGINT ( 0xee, 0xcb, 0x7b, 0xfd ),
				32 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x01, 0xdd ),
				9 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff ),
				32 );
	bigint_max_set_bit_ok ( BIGINT ( 0x32, 0x39, 0x96, 0x52, 0x10, 0x67,
					 0x7e, 0x32, 0xfc, 0x4e, 0x56, 0xc3,
					 0x68, 0x18, 0x76, 0x1a, 0xac, 0x0e,
					 0x93, 0xee, 0x55, 0xc5, 0x6e ),
				182 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0xc8, 0xe6, 0x59 ),
				24 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff ),
				184 );
	bigint_max_set_bit_ok ( BIGINT ( 0xcd, 0xb3, 0x22, 0x30, 0xdd, 0xa7,
					 0xff, 0x37, 0xbf, 0xe3, 0x38, 0xf7,
					 0xe1, 0x41, 0x73, 0xea, 0x3a, 0xfc,
					 0x78, 0x9e, 0xfb, 0x4f, 0x85, 0xdc,
					 0x1c, 0x40, 0x89, 0x6e, 0xda, 0xf9,
					 0x9d, 0x6d, 0x12, 0x97, 0xb1, 0x80,
					 0x2a, 0xeb, 0x91, 0xce, 0x3b, 0x83,
					 0xb8, 0xa5, 0x3d, 0xce, 0x46, 0x56,
					 0xb7, 0xd1, 0x28, 0xbc, 0x93, 0x4e,
					 0x8c, 0x29, 0x6d, 0x2c, 0xcc, 0x58,
					 0x49, 0x2f, 0x37, 0xa0, 0x08, 0x37,
					 0x86, 0xdd, 0x38, 0x21, 0xa7, 0x57,
					 0x37, 0xe3, 0xc5, 0xcc, 0x50, 0x11,
					 0x1a, 0xe4, 0xea, 0xe7, 0x4d, 0x3c,
					 0x37, 0x65, 0x78, 0xd1, 0xf6, 0xc3,
					 0x94, 0x46, 0xd4, 0x0e, 0xd3, 0x9a,
					 0x21, 0x8b, 0xa6, 0x54, 0xc0, 0xd2,
					 0x88, 0x07, 0x24, 0xbf, 0x7d, 0x31,
					 0xfd, 0x15, 0xa8, 0x92, 0x65, 0xe1,
					 0x8d, 0xed, 0x70, 0x7b, 0x68, 0x0f,
					 0xcc, 0x13, 0xb9, 0xb2, 0xdd, 0x3c,
					 0x6a, 0x52 ),
				1024 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x66, 0xd9,
					 0x91, 0x18, 0x6e, 0xd3, 0xff, 0x9b,
					 0xdf, 0xf1, 0x9c, 0x7b, 0xf0, 0xa0,
					 0xb9, 0xf5 ),
				127 );
	bigint_max_set_bit_ok ( BIGINT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					 0x00, 0x00 ),
				0 );
	bigint_max_set_bit_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					 0xff, 0xff ),
				1024 );
	bigint_multiply_ok ( BIGINT ( 0xf0 ),
			     BIGINT ( 0xeb ),
			     BIGINT ( 0xdc, 0x50 ) );
	bigint_multiply_ok ( BIGINT ( 0xd7, 0x16 ),
			     BIGINT ( 0x88, 0xfb ),
			     BIGINT ( 0x73, 0x16, 0x92, 0x92 ) );
	bigint_multiply_ok ( BIGINT ( 0xfe, 0xed, 0x1d ),
			     BIGINT ( 0x69, 0x9c, 0x03 ),
			     BIGINT ( 0x69, 0x2a, 0x9c, 0x5f, 0x73, 0x57 ) );
	bigint_multiply_ok ( BIGINT ( 0x96, 0xe9, 0x6f, 0x81 ),
			     BIGINT ( 0x67, 0x3c, 0x5a, 0x16 ),
			     BIGINT ( 0x3c, 0xdb, 0x7f, 0xae, 0x12, 0x7e,
				      0xef, 0x16 ) );
	bigint_multiply_ok ( BIGINT ( 0xe8, 0x08, 0x0b, 0xe9, 0x29, 0x36,
				      0xea, 0x51, 0x1d, 0x75, 0x1a, 0xd5,
				      0xba, 0xc6, 0xa0, 0xf3, 0x48, 0x5c,
				      0xdf, 0x42, 0xdf, 0x28, 0x38 ),
			     BIGINT ( 0x22, 0x07, 0x41, 0x54, 0x4e, 0xf9,
				      0x90, 0xa8, 0xaf, 0xba, 0xf6, 0xb0,
				      0x35, 0x7e, 0x98, 0xef, 0x2c, 0x31,
				      0xc9, 0xa7, 0x25, 0x74, 0x8d ),
			     BIGINT ( 0x1e, 0xd7, 0xa5, 0x03, 0xc0, 0x18,
				      0x2e, 0x29, 0xb1, 0x3e, 0x96, 0x71,
				      0x90, 0xa5, 0x6d, 0x43, 0x58, 0xf7,
				      0x22, 0x80, 0x0b, 0x21, 0xc6, 0x70,
				      0x90, 0x1c, 0xa8, 0x85, 0x87, 0xaf,
				      0xd7, 0xdd, 0x27, 0x69, 0xaf, 0x20,
				      0xa0, 0x2d, 0x43, 0x5d, 0xda, 0xba,
				      0x4b, 0x3a, 0x86, 0xd8 ) );
	bigint_multiply_ok ( BIGINT ( 0xa2, 0x0f, 0xc6, 0x08, 0x0a, 0x01,
				      0x19, 0x42, 0x0e, 0xaa, 0x5c, 0xae,
				      0x4f, 0x4e, 0xb0, 0xad, 0xb2, 0xe8,
				      0xee, 0xd5, 0x65, 0xec, 0x5a, 0xda,
				      0xc0, 0xba, 0x78, 0xa8, 0x0f, 0x15,
				      0x39, 0xd7, 0x7a, 0x10, 0xc2, 0xa7,
				      0xec, 0x44, 0xac, 0xad, 0x39, 0x04,
				      0x2e, 0x66, 0x54, 0x70, 0x57, 0xee,
				      0xf6, 0x97, 0x19, 0x71, 0x16, 0xf9,
				      0xbb, 0x2e, 0x84, 0x09, 0x6e, 0x9a,
				      0x3b, 0x16, 0xb2, 0x65, 0x74, 0x50,
				      0x19, 0xd1, 0xe9, 0x95, 0xa0, 0x7b,
				      0x33, 0xb5, 0xac, 0x7c, 0x9e, 0xd4,
				      0x68, 0x0d, 0xc9, 0xe4, 0x03, 0x86,
				      0x1a, 0xa3, 0x42, 0x33, 0x28, 0x14,
				      0x12, 0x7d, 0x5a, 0xd9, 0x30, 0x18,
				      0x0a, 0xf4, 0x0c, 0x96, 0x58, 0xc9,
				      0xb5, 0x37, 0xdb, 0x49, 0xdc, 0x01,
				      0x4a, 0xcb, 0x6d, 0x87, 0x52, 0xf6,
				      0xae, 0xa7, 0x71, 0x31, 0x9a, 0x1a,
				      0xe2, 0x1c, 0x87, 0x51, 0xc9, 0xeb,
				      0x70, 0x71 ),
			     BIGINT ( 0x7c, 0xdd, 0x2f, 0x5d, 0x27, 0xfe,
				      0xca, 0x70, 0x96, 0xc3, 0xb1, 0x1f,
				      0xac, 0xa9, 0x3a, 0xdc, 0xcd, 0xbc,
				      0x58, 0xb4, 0xde, 0xe7, 0xe5, 0x34,
				      0x1a, 0xc0, 0xb9, 0x46, 0xf7, 0x52,
				      0x76, 0x23, 0xe8, 0xe9, 0x92, 0xa1,
				      0x86, 0x3c, 0x6f, 0xf1, 0x22, 0xf4,
				      0x72, 0xb1, 0xde, 0xd3, 0x8f, 0x11,
				      0x9e, 0x52, 0xe5, 0x81, 0x54, 0xe9,
				      0xa7, 0x72, 0x3f, 0x3e, 0xa0, 0x80,
				      0xbb, 0xae, 0x0e, 0x30, 0x6a, 0x11,
				      0x91, 0x11, 0x3b, 0x3f, 0x44, 0x1f,
				      0x8d, 0x4d, 0xea, 0xdd, 0x09, 0x95,
				      0x9d, 0x02, 0xa6, 0x6d, 0x3b, 0x08,
				      0x40, 0x8d, 0xb4, 0x4b, 0x05, 0x74,
				      0x8c, 0x1f, 0xaa, 0x61, 0x6f, 0x0e,
				      0xcc, 0xcf, 0xe0, 0x81, 0x03, 0xe4,
				      0x9b, 0x11, 0xd9, 0xab, 0xf3, 0x24,
				      0xe2, 0x3b, 0xe0, 0x05, 0x60, 0x65,
				      0x16, 0xc6, 0x2e, 0x83, 0xa0, 0x98,
				      0x8e, 0x11, 0x05, 0x00, 0xe4, 0x3f,
				      0x7e, 0x65 ),
			     BIGINT ( 0x4f, 0x0b, 0xa9, 0x85, 0xb8, 0x31,
				      0x48, 0xea, 0x11, 0x44, 0xaf, 0x2d,
				      0xed, 0x1a, 0x76, 0x45, 0xac, 0x87,
				      0x0c, 0xf3, 0xd7, 0xc4, 0x8e, 0x5c,
				      0xd7, 0xdf, 0x28, 0x74, 0xa6, 0x40,
				      0xe4, 0x6b, 0x5b, 0x19, 0x36, 0x37,
				      0x9c, 0xcd, 0x43, 0x76, 0x15, 0x00,
				      0x5d, 0x23, 0xa2, 0x8a, 0x53, 0x25,
				      0xbf, 0x18, 0xda, 0xe6, 0x09, 0xdf,
				      0xaa, 0xeb, 0x9a, 0x82, 0x01, 0x14,
				      0x2b, 0x20, 0x2b, 0xb6, 0x22, 0x62,
				      0x6b, 0xcc, 0xd4, 0xc9, 0x02, 0x67,
				      0x95, 0x43, 0x75, 0x4e, 0x97, 0x4e,
				      0xec, 0x04, 0xde, 0x29, 0x0a, 0xef,
				      0xf7, 0xc1, 0x72, 0x8c, 0x64, 0x38,
				      0x16, 0x47, 0x9f, 0x16, 0x0c, 0xa5,
				      0x79, 0x6b, 0xea, 0x2e, 0x4c, 0x3d,
				      0x0c, 0xe6, 0x57, 0x51, 0x65, 0xa5,
				      0x3b, 0xca, 0xae, 0x54, 0x0c, 0x67,
				      0xf8, 0x23, 0x00, 0xc9, 0x8d, 0xe6,
				      0x16, 0x91, 0x19, 0xb3, 0x5b, 0x68,
				      0x7b, 0xf2, 0xe2, 0x5d, 0x69, 0x48,
				      0x3f, 0x2b, 0xa0, 0x4f, 0x7c, 0x3c,
				      0x26, 0xf9, 0xd9, 0xfd, 0x3d, 0x5d,
				      0xd6, 0x05, 0x00, 0xd8, 0xdf, 0x5a,
				      0x56, 0x8f, 0x16, 0x68, 0x4f, 0x15,
				      0x19, 0x9d, 0xd7, 0x11, 0x51, 0x7d,
				      0x73, 0x5c, 0xd4, 0xd5, 0xb4, 0xc7,
				      0x42, 0xe3, 0xee, 0xf1, 0x67, 0xd6,
				      0x69, 0x72, 0x04, 0x4b, 0x88, 0x3d,
				      0x05, 0xd8, 0x1e, 0x50, 0xcb, 0xce,
				      0x39, 0x19, 0x42, 0xb6, 0xa7, 0xf3,
				      0xba, 0x78, 0x90, 0xd2, 0x09, 0x05,
				      0x87, 0xf8, 0xc0, 0x9c, 0x47, 0xff,
				      0xbf, 0xaa, 0x21, 0x8d, 0x81, 0x86,
				      0xcd, 0x58, 0xdf, 0x30, 0xf1, 0xd1,
				      0x60, 0x53, 0x85, 0x40, 0xbf, 0x14,
				      0x3e, 0xdc, 0x9e, 0x9e, 0xc4, 0xc7,
				      0x48, 0xa0, 0x83, 0xe0, 0x99, 0x8b,
				      0x43, 0xf8, 0x52, 0x8a, 0x15, 0x88,
				      0x89, 0x83, 0x7d, 0x71, 0xbb, 0x62,
				      0x12, 0x7a, 0x23, 0x85, 0x3a, 0xbb,
				      0xdb, 0x09, 0xfa, 0x95 ) );
	bigint_multiply_ok ( BIGINT ( 0xff ),
			     BIGINT ( 0xff ),
			     BIGINT ( 0xfe, 0x01 ) );
	bigint_multiply_ok ( BIGINT ( 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff ),
			     BIGINT ( 0xff, 0xfe, 0x00, 0x01 ) );
	bigint_multiply_ok ( BIGINT ( 0xff, 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xfe, 0x00, 0x00, 0x01 ) );
	bigint_multiply_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00,
				      0x00, 0x01 ) );
	bigint_multiply_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xfe, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x01 ) );
	bigint_multiply_ok ( BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff ),
			     BIGINT ( 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				      0xff, 0xfe, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x00, 0x00, 0x01 ) );
	bigint_mod_multiply_ok ( BIGINT ( 0x37 ),
				 BIGINT ( 0x67 ),
				 BIGINT ( 0x3f ),
				 BIGINT ( 0x3a ) );
	bigint_mod_multiply_ok ( BIGINT ( 0x45, 0x94 ),
				 BIGINT ( 0xbd, 0xd2 ),
				 BIGINT ( 0xca, 0xc7 ),
				 BIGINT ( 0xac, 0xc1 ) );
	bigint_mod_multiply_ok ( BIGINT ( 0x8e, 0xcd, 0x74 ),
				 BIGINT ( 0xe2, 0xf1, 0xea ),
				 BIGINT ( 0x59, 0x51, 0x53 ),
				 BIGINT ( 0x22, 0xdd, 0x1c ) );
	bigint_mod_multiply_ok ( BIGINT ( 0xc2, 0xa8, 0x40, 0x6f ),
				 BIGINT ( 0x29, 0xd7, 0xf4, 0x77 ),
				 BIGINT ( 0xbb, 0xbd, 0xdb, 0x3d ),
				 BIGINT ( 0x8f, 0x39, 0xd0, 0x47 ) );
	bigint_mod_multiply_ok ( BIGINT ( 0x2e, 0xcb, 0x74, 0x7c, 0x64, 0x60,
					  0xb3, 0x44, 0xf3, 0x23, 0x49, 0x4a,
					  0xc6, 0xb6, 0xbf, 0xea, 0x80, 0xd8,
					  0x34, 0xc5, 0x54, 0x22, 0x09 ),
				 BIGINT ( 0x61, 0x2c, 0x5a, 0xc5, 0xde, 0x07,
					  0x65, 0x8e, 0xab, 0x88, 0x1a, 0x2e,
					  0x7a, 0x95, 0x17, 0xe3, 0x3b, 0x17,
					  0xe4, 0x21, 0xb0, 0xb4, 0x57 ),
				 BIGINT ( 0x8e, 0x46, 0xa5, 0x87, 0x7b, 0x7f,
					  0xc4, 0xd7, 0x31, 0xb1, 0x94, 0xe7,
					  0xe7, 0x1c, 0x7e, 0x7a, 0xc2, 0x6c,
					  0xce, 0xcb, 0xc8, 0x5d, 0x70 ),
				 BIGINT ( 0x1e, 0xd1, 0x5b, 0x3d, 0x1d, 0x17,
					  0x7c, 0x31, 0x95, 0x13, 0x1b, 0xd8,
					  0xee, 0x0a, 0xb0, 0xe1, 0x5b, 0x13,
					  0xad, 0x83, 0xe9, 0xf8, 0x7f ) );
	bigint_mod_multiply_ok ( BIGINT ( 0x56, 0x37, 0xab, 0x07, 0x8b, 0x25,
					  0xa7, 0xc2, 0x50, 0x30, 0x43, 0xfc,
					  0x63, 0x8b, 0xdf, 0x84, 0x68, 0x85,
					  0xca, 0xce, 0x44, 0x5c, 0xb1, 0x14,
					  0xa4, 0xb5, 0xba, 0x43, 0xe0, 0x31,
					  0x45, 0x6b, 0x82, 0xa9, 0x0b, 0x9e,
					  0x3a, 0xb0, 0x39, 0x09, 0x2a, 0x68,
					  0x2e, 0x0f, 0x09, 0x54, 0x47, 0x04,
					  0xdb, 0xcf, 0x4a, 0x3a, 0x2d, 0x7b,
					  0x7d, 0x50, 0xa4, 0xc5, 0xeb, 0x13,
					  0xdd, 0x49, 0x61, 0x7d, 0x18, 0xa1,
					  0x0d, 0x6b, 0x58, 0xba, 0x9f, 0x7c,
					  0x81, 0x34, 0x9e, 0xf9, 0x9c, 0x9e,
					  0x28, 0xa8, 0x1c, 0x15, 0x16, 0x20,
					  0x3c, 0x0a, 0xb1, 0x11, 0x06, 0x93,
					  0xbc, 0xd8, 0x4e, 0x49, 0xae, 0x7b,
					  0xb4, 0x02, 0x8b, 0x1c, 0x5b, 0x18,
					  0xb4, 0xac, 0x7f, 0xdd, 0x70, 0xef,
					  0x87, 0xac, 0x1b, 0xac, 0x25, 0xa3,
					  0xc9, 0xa7, 0x3a, 0xc5, 0x76, 0x68,
					  0x09, 0x1f, 0xa1, 0x48, 0x53, 0xb6,
					  0x13, 0xac ),
				 BIGINT ( 0xef, 0x5c, 0x1f, 0x1a, 0x44, 0x64,
					  0x66, 0xcf, 0xdd, 0x3f, 0x0b, 0x27,
					  0x81, 0xa7, 0x91, 0x7a, 0x35, 0x7b,
					  0x0f, 0x46, 0x5e, 0xca, 0xbf, 0xf8,
					  0x50, 0x5e, 0x99, 0x7c, 0xc6, 0x64,
					  0x43, 0x00, 0x9f, 0xb2, 0xda, 0xfa,
					  0x42, 0x15, 0x9c, 0xa3, 0xd6, 0xc8,
					  0x64, 0xa7, 0x65, 0x4a, 0x98, 0xf7,
					  0xb3, 0x96, 0x5f, 0x42, 0xf9, 0x73,
					  0xe1, 0x75, 0xc3, 0xc4, 0x0b, 0x5d,
					  0x5f, 0xf3, 0x04, 0x8a, 0xee, 0x59,
					  0xa6, 0x1b, 0x06, 0x38, 0x0b, 0xa2,
					  0x9f, 0xb4, 0x4f, 0x6d, 0x50, 0x5e,
					  0x37, 0x37, 0x21, 0x83, 0x9d, 0xa3,
					  0x12, 0x16, 0x4d, 0xab, 0x36, 0x51,
					  0x21, 0xb1, 0x74, 0x66, 0x40, 0x9a,
					  0xd3, 0x72, 0xcc, 0x18, 0x40, 0x53,
					  0x89, 0xff, 0xd7, 0x00, 0x8d, 0x7e,
					  0x93, 0x81, 0xdb, 0x29, 0xb6, 0xd7,
					  0x23, 0x2b, 0x67, 0x2f, 0x11, 0x98,
					  0x49, 0x87, 0x2f, 0x46, 0xb7, 0x33,
					  0x6d, 0x12 ),
				 BIGINT ( 0x67, 0x7a, 0x17, 0x6a, 0xd2, 0xf8,
					  0x49, 0xfb, 0x7c, 0x95, 0x25, 0x54,
					  0xf0, 0xab, 0x5b, 0xb3, 0x0e, 0x01,
					  0xab, 0xd3, 0x65, 0x6f, 0x7e, 0x18,
					  0x05, 0xed, 0x9b, 0xc4, 0x90, 0x6c,
					  0xd0, 0x6d, 0x94, 0x79, 0x28, 0xd6,
					  0x24, 0x77, 0x9a, 0x08, 0xd2, 0x2f,
					  0x7c, 0x2d, 0xa0, 0x0c, 0x14, 0xbe,
					  0x7b, 0xee, 0x9e, 0x48, 0x88, 0x3c,
					  0x8f, 0x9f, 0xb9, 0x7a, 0xcb, 0x98,
					  0x76, 0x61, 0x0d, 0xee, 0xa2, 0x42,
					  0x67, 0x1b, 0x2c, 0x42, 0x8f, 0x41,
					  0xcc, 0x78, 0xba, 0xba, 0xaa, 0xa2,
					  0x92, 0xb0, 0x6e, 0x0c, 0x4e, 0xe1,
					  0xa5, 0x13, 0x7d, 0x8a, 0x8f, 0x81,
					  0x95, 0x8a, 0xdf, 0x57, 0x93, 0x88,
					  0x27, 0x4f, 0x1a, 0x59, 0xa4, 0x74,
					  0x22, 0xa9, 0x78, 0xe5, 0xed, 0xb1,
					  0x09, 0x26, 0x59, 0xde, 0x88, 0x21,
					  0x8d, 0xa2, 0xa8, 0x58, 0x10, 0x7b,
					  0x65, 0x96, 0xbf, 0x69, 0x3b, 0xc5,
					  0x55, 0xf8 ),
				 BIGINT ( 0x15, 0xf7, 0x00, 0xeb, 0xc7, 0x5a,
					  0x6f, 0xf0, 0x50, 0xf3, 0x21, 0x8a,
					  0x8e, 0xa6, 0xf6, 0x67, 0x56, 0x7d,
					  0x07, 0x45, 0x89, 0xdb, 0xd7, 0x7e,
					  0x9e, 0x28, 0x7f, 0xfb, 0xed, 0xca,
					  0x2c, 0xbf, 0x47, 0x77, 0x99, 0x95,
					  0xf3, 0xd6, 0x9d, 0xc5, 0x57, 0x81,
					  0x7f, 0x97, 0xf2, 0x6b, 0x24, 0xee,
					  0xce, 0xc5, 0x9b, 0xe6, 0x42, 0x2d,
					  0x37, 0xb7, 0xeb, 0x3d, 0xb5, 0xf7,
					  0x1e, 0x86, 0xc2, 0x40, 0x44, 0xc9,
					  0x85, 0x5a, 0x1a, 0xc0, 0xac, 0x9e,
					  0x78, 0x69, 0x00, 0x7b, 0x93, 0x65,
					  0xd7, 0x7f, 0x0c, 0xd6, 0xba, 0x4f,
					  0x06, 0x00, 0x61, 0x05, 0xb2, 0x44,
					  0xb4, 0xe7, 0xbb, 0x3b, 0x96, 0xb0,
					  0x6d, 0xe8, 0x43, 0xd2, 0x03, 0xb7,
					  0x0a, 0xc4, 0x6d, 0x30, 0xd8, 0xd5,
					  0xe6, 0x54, 0x65, 0xdd, 0xa9, 0x1b,
					  0x50, 0xc0, 0xb9, 0x95, 0xb0, 0x7d,
					  0x7c, 0xca, 0x63, 0xf8, 0x72, 0xbe,
					  0x3b, 0x00 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xcd ),
			    BIGINT ( 0xbb ),
			    BIGINT ( 0x25 ),
			    BIGINT ( 0xab ) );
	bigint_mod_exp_ok ( BIGINT ( 0xc4 ),
			    BIGINT ( 0xe9 ),
			    BIGINT ( 0x02, 0x4c ),
			    BIGINT ( 0x7e ) );
	bigint_mod_exp_ok ( BIGINT ( 0xcb ),
			    BIGINT ( 0xde ),
			    BIGINT ( 0xbd, 0x73, 0xbf ),
			    BIGINT ( 0x17 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x17 ),
			    BIGINT ( 0xb9 ),
			    BIGINT ( 0x39, 0x68, 0xba, 0x7d ),
			    BIGINT ( 0x17 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x2e ),
			    BIGINT ( 0xb7 ),
			    BIGINT ( 0x39, 0x07, 0x1b, 0x49, 0x5b, 0xea,
				     0xf2, 0x61, 0x75, 0x94, 0x60, 0x86,
				     0x73, 0xd0, 0xeb, 0x11, 0x08, 0x19,
				     0x90, 0x19, 0xe0, 0xed, 0x2a ),
			    BIGINT ( 0x19 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x59 ),
			    BIGINT ( 0xce ),
			    BIGINT ( 0xdf, 0xbc, 0x0d, 0x0c, 0x09, 0xeb,
				     0xf8, 0xcf, 0xdb, 0xb6, 0x00, 0xa3,
				     0x9e, 0xc3, 0x6c, 0x8d, 0xf1, 0xc3,
				     0x03, 0x36, 0xaa, 0xd4, 0x22, 0x7c,
				     0x20, 0x7b, 0xa9, 0x9a, 0x01, 0xe4,
				     0xf2, 0x50, 0x42, 0x29, 0x68, 0x7a,
				     0xa6, 0x2c, 0xdf, 0xb6, 0x51, 0xa9,
				     0x73, 0x10, 0x98, 0x37, 0x69, 0xb3,
				     0x21, 0x49, 0x6d, 0xcc, 0x80, 0xfa,
				     0x7e, 0x12, 0xe4, 0x9c, 0xc2, 0xbb,
				     0xe3, 0xa3, 0x10, 0x3f, 0xba, 0x99,
				     0x22, 0x79, 0x71, 0x39, 0x96, 0x7b,
				     0x1a, 0x89, 0xdc, 0xda, 0x43, 0x52,
				     0x50, 0x7b, 0xe3, 0x8c, 0xd3, 0xc0,
				     0xf5, 0x7d, 0xfc, 0x80, 0x71, 0x6e,
				     0xaf, 0x5c, 0xd0, 0x14, 0xc0, 0x60,
				     0x24, 0xa8, 0x9a, 0x8a, 0x54, 0x4a,
				     0x6f, 0x42, 0x7a, 0x14, 0x14, 0x25,
				     0xd5, 0x22, 0x08, 0x8f, 0xd9, 0xdb,
				     0xd4, 0x0f, 0x14, 0xf4, 0x3b, 0x26,
				     0x0e, 0xb6, 0x72, 0xd7, 0x03, 0xd5,
				     0xf0, 0x0e ),
			    BIGINT ( 0xa9 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x7f, 0x30 ),
			    BIGINT ( 0x73, 0x74 ),
			    BIGINT ( 0x75 ),
			    BIGINT ( 0x4b, 0xe8 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x04, 0x6c ),
			    BIGINT ( 0x99, 0x04 ),
			    BIGINT ( 0x33, 0xd2 ),
			    BIGINT ( 0x86, 0x74 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xca, 0x88 ),
			    BIGINT ( 0xdc, 0x60 ),
			    BIGINT ( 0x7e, 0x76, 0x79 ),
			    BIGINT ( 0x42, 0x40 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x68, 0x97 ),
			    BIGINT ( 0x52, 0x8b ),
			    BIGINT ( 0x4f, 0x7f, 0xe7, 0xda ),
			    BIGINT ( 0x22, 0x77 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xbd, 0x14 ),
			    BIGINT ( 0x9e, 0xfc ),
			    BIGINT ( 0x23, 0xf7, 0xd0, 0xa1, 0x9e, 0x9b,
				     0x05, 0xd2, 0x44, 0x24, 0x4f, 0x3f,
				     0x83, 0xcc, 0x49, 0x70, 0xa5, 0x0d,
				     0xfc, 0xa7, 0x43, 0xf3, 0x3e ),
			    BIGINT ( 0x1a, 0xc8 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x46, 0x3e ),
			    BIGINT ( 0xb8, 0xde ),
			    BIGINT ( 0xa9, 0xc0, 0xdc, 0x45, 0x65, 0x0d,
				     0xa5, 0x56, 0x70, 0x4c, 0xf1, 0xda,
				     0xab, 0x64, 0xc2, 0x04, 0xf6, 0x32,
				     0x20, 0x68, 0x31, 0x5f, 0x9a, 0x00,
				     0x0f, 0x7b, 0x24, 0x33, 0xdf, 0xaf,
				     0xfe, 0x03, 0x1e, 0x4a, 0xa1, 0xf8,
				     0x45, 0x8d, 0x5a, 0x7d, 0x12, 0x58,
				     0x00, 0x6d, 0xba, 0x79, 0x9f, 0xe1,
				     0xa1, 0xfc, 0x1f, 0xb9, 0xf3, 0xa7,
				     0x07, 0xf5, 0xfe, 0xd6, 0xa1, 0xba,
				     0xda, 0x63, 0xef, 0x39, 0x8e, 0xb7,
				     0x48, 0xa8, 0x81, 0x86, 0xb1, 0x22,
				     0x14, 0x9f, 0x9e, 0xac, 0x69, 0xf7,
				     0xae, 0x1f, 0xf2, 0x99, 0x41, 0xb7,
				     0x37, 0xa7, 0xbc, 0x42, 0xf2, 0x45,
				     0x43, 0xf2, 0x2a, 0xef, 0xc2, 0x83,
				     0xd5, 0x32, 0x6e, 0xfa, 0x49, 0x1c,
				     0x94, 0x9c, 0xc2, 0xc5, 0xad, 0x28,
				     0x53, 0x1c, 0x11, 0xc4, 0x1c, 0x78,
				     0x8f, 0x13, 0xdc, 0xb3, 0x2a, 0x63,
				     0xfd, 0x1f, 0x89, 0x9b, 0x0c, 0x31,
				     0x92, 0x73 ),
			    BIGINT ( 0x7b, 0x8a ) );
	bigint_mod_exp_ok ( BIGINT ( 0xf3, 0xc3, 0xab ),
			    BIGINT ( 0xd0, 0x7e, 0xd0 ),
			    BIGINT ( 0xf6 ),
			    BIGINT ( 0x1f, 0xb3, 0x09 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x13, 0xec, 0xf6 ),
			    BIGINT ( 0x87, 0x1a, 0x9a ),
			    BIGINT ( 0x03, 0xf3 ),
			    BIGINT ( 0x15, 0xe9, 0x8e ) );
	bigint_mod_exp_ok ( BIGINT ( 0x5a, 0x96, 0xe5 ),
			    BIGINT ( 0x56, 0x4a, 0xd1 ),
			    BIGINT ( 0x89, 0x62, 0x8e ),
			    BIGINT ( 0x34, 0xb8, 0xaa ) );
	bigint_mod_exp_ok ( BIGINT ( 0x84, 0x7c, 0xbd ),
			    BIGINT ( 0x3c, 0x80, 0x0a ),
			    BIGINT ( 0x5e, 0x52, 0x9d, 0xba ),
			    BIGINT ( 0x04, 0xcb, 0x4f ) );
	bigint_mod_exp_ok ( BIGINT ( 0x50, 0x01, 0x51 ),
			    BIGINT ( 0x02, 0xe6, 0x96 ),
			    BIGINT ( 0x34, 0x0c, 0x7e, 0xbf, 0x27, 0x23,
				     0x46, 0x92, 0x1c, 0xca, 0x91, 0xab,
				     0x50, 0x2c, 0x3a, 0x64, 0xc8, 0x4a,
				     0x75, 0xd6, 0xe2, 0xde, 0x31 ),
			    BIGINT ( 0x02, 0x16, 0x05 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x5e, 0x47, 0xd8 ),
			    BIGINT ( 0x26, 0xd1, 0xb6 ),
			    BIGINT ( 0x49, 0x61, 0x84, 0x7a, 0xa9, 0xfb,
				     0x93, 0x45, 0xe4, 0xfa, 0x53, 0x60,
				     0x73, 0x98, 0x5a, 0x17, 0xe7, 0x77,
				     0x2d, 0xcd, 0x97, 0xf4, 0xc0, 0x34,
				     0x46, 0xfa, 0xbd, 0x21, 0xdf, 0xa5,
				     0xa0, 0x12, 0x38, 0x7c, 0xbd, 0xd9,
				     0xcd, 0xbc, 0xde, 0x29, 0xa5, 0x13,
				     0xa8, 0xf0, 0xf6, 0x88, 0xc6, 0x31,
				     0xed, 0x90, 0x19, 0x11, 0x7d, 0xe1,
				     0x0e, 0x81, 0x98, 0x8e, 0x98, 0x86,
				     0xde, 0x2a, 0x4c, 0xad, 0xff, 0x57,
				     0x12, 0xbc, 0x4b, 0xaf, 0x21, 0xde,
				     0xca, 0x3a, 0x25, 0xd7, 0x98, 0xe3,
				     0x25, 0xbc, 0x17, 0x74, 0x0b, 0x9c,
				     0x53, 0xe1, 0x1a, 0xec, 0x9a, 0x5a,
				     0xdc, 0x68, 0xdf, 0xad, 0xd6, 0x71,
				     0x6b, 0x5b, 0x8b, 0x85, 0xbb, 0xe5,
				     0xd5, 0x14, 0x4c, 0x30, 0x27, 0x68,
				     0xd1, 0xf7, 0x58, 0x34, 0x4c, 0xe1,
				     0x71, 0xde, 0x7b, 0x8d, 0xa2, 0xe6,
				     0x0a, 0x44, 0x22, 0x26, 0x5a, 0x70,
				     0xbb, 0x68 ),
			    BIGINT ( 0x18, 0x36, 0x96 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xc7, 0x4a, 0xf0, 0x48 ),
			    BIGINT ( 0x5d, 0x27, 0x07, 0x54 ),
			    BIGINT ( 0x4a ),
			    BIGINT ( 0x48, 0x68, 0x7b, 0xe0 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xb4, 0x89, 0xc9, 0x5b ),
			    BIGINT ( 0x7c, 0xd7, 0xc7, 0xff ),
			    BIGINT ( 0xc6, 0x9c ),
			    BIGINT ( 0x0b, 0x2d, 0xf8, 0xf7 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xea, 0x72, 0x43, 0xfe ),
			    BIGINT ( 0xfc, 0x57, 0x2d, 0x47 ),
			    BIGINT ( 0x60, 0x01, 0x2c ),
			    BIGINT ( 0x12, 0x01, 0xe3, 0xf5 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x81, 0x7f, 0x27, 0x94 ),
			    BIGINT ( 0x17, 0x21, 0x67, 0xab ),
			    BIGINT ( 0x50, 0x19, 0x12, 0x52 ),
			    BIGINT ( 0x05, 0x17, 0x6b, 0x13 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x38, 0xab, 0xd4, 0xec ),
			    BIGINT ( 0x0c, 0x2a, 0x56, 0x38 ),
			    BIGINT ( 0x2f, 0x85, 0x85, 0x57, 0xf6, 0xde,
				     0x24, 0xb4, 0x28, 0x3c, 0x5a, 0x3c,
				     0x0b, 0x12, 0x85, 0x85, 0x85, 0x98,
				     0x46, 0x5b, 0x9c, 0x52, 0x3a ),
			    BIGINT ( 0x02, 0xe6, 0x6a, 0x70 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xa6, 0x35, 0xc0, 0x6f ),
			    BIGINT ( 0x23, 0xac, 0x78, 0x72 ),
			    BIGINT ( 0x6a, 0x07, 0x80, 0xbf, 0x1b, 0xa5,
				     0xf8, 0x0b, 0x90, 0x06, 0xa4, 0xa5,
				     0x44, 0x13, 0xba, 0x4b, 0xb3, 0xce,
				     0x9f, 0x55, 0x42, 0x56, 0xc3, 0x30,
				     0x82, 0x85, 0x5a, 0x3b, 0xae, 0x88,
				     0x92, 0x4e, 0x3c, 0x37, 0xf6, 0x80,
				     0x4c, 0x03, 0x3c, 0x1e, 0x2c, 0x17,
				     0xef, 0x9d, 0xd7, 0x6f, 0xdc, 0xbb,
				     0x42, 0x42, 0xa1, 0x7f, 0x97, 0x66,
				     0xcd, 0xc8, 0x8a, 0x7c, 0xc6, 0x70,
				     0x61, 0x54, 0x82, 0xd0, 0xd0, 0x8b,
				     0xd5, 0x4f, 0x57, 0x7b, 0x8e, 0xab,
				     0xdc, 0xbf, 0x8e, 0x85, 0x94, 0x83,
				     0x8a, 0xb3, 0x72, 0x69, 0x2d, 0x51,
				     0xdd, 0x86, 0x1e, 0x58, 0xb8, 0x00,
				     0xe2, 0x5e, 0xa7, 0xef, 0x6a, 0x6a,
				     0xb0, 0x10, 0x3d, 0x53, 0xfe, 0x23,
				     0x51, 0xc0, 0x51, 0xed, 0x1f, 0x02,
				     0x4b, 0x73, 0x17, 0x59, 0xfa, 0xb9,
				     0xa8, 0x05, 0xa7, 0x79, 0xc3, 0xc9,
				     0x4c, 0x2d, 0x58, 0x59, 0x10, 0x99,
				     0x71, 0xe6 ),
			    BIGINT ( 0x01, 0x63, 0xd0, 0x07 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xff, 0x2a, 0x37, 0x04, 0xd4, 0x08,
				     0x9f, 0xf5, 0xac, 0x29, 0x7f, 0x4b,
				     0x93, 0x86, 0x02, 0x26, 0xac, 0x29,
				     0xa8, 0xf9, 0x77, 0x91, 0x20 ),
			    BIGINT ( 0x2c, 0xb2, 0xe2, 0x1f, 0x4b, 0x97,
				     0xaa, 0x3b, 0xd1, 0x36, 0xb0, 0x40,
				     0x8b, 0x1c, 0x19, 0xa2, 0xea, 0xc8,
				     0xc6, 0x4e, 0x2a, 0x66, 0x50 ),
			    BIGINT ( 0x97 ),
			    BIGINT ( 0x04, 0x22, 0x44, 0xe2, 0x14, 0x54,
				     0x6c, 0x5a, 0xba, 0x1b, 0x39, 0xb7,
				     0xaa, 0x06, 0xcf, 0x2b, 0xc8, 0x7e,
				     0xc0, 0xe0, 0x70, 0xf2, 0x90 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xcd, 0xf3, 0xf7, 0x50, 0x13, 0x39,
				     0x13, 0x4a, 0x56, 0xc5, 0xb8, 0xa6,
				     0x42, 0x2d, 0x40, 0x5e, 0x07, 0xf2,
				     0x92, 0x2a, 0x51, 0x87, 0x20 ),
			    BIGINT ( 0x93, 0x1a, 0x28, 0xbb, 0x69, 0x4f,
				     0x31, 0x01, 0xe0, 0x88, 0x8a, 0x4c,
				     0x4f, 0x9b, 0xda, 0xf6, 0x4e, 0xf3,
				     0x11, 0xe7, 0x35, 0xa1, 0xfb ),
			    BIGINT ( 0x66, 0x69 ),
			    BIGINT ( 0x7a, 0x5a, 0x9b, 0x84, 0x72, 0x8f,
				     0x57, 0x31, 0xb4, 0x34, 0x70, 0x18,
				     0x77, 0xa6, 0x43, 0xa9, 0x51, 0x69,
				     0x07, 0x3e, 0xf6, 0x68, 0x82 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xdd, 0x4c, 0x85, 0xcb, 0x3f, 0x45,
				     0x61, 0xe0, 0x58, 0x1e, 0xad, 0xd3,
				     0x6b, 0xef, 0x82, 0x53, 0x4a, 0x16,
				     0x1a, 0xf0, 0x09, 0x82, 0x74 ),
			    BIGINT ( 0xd2, 0xa2, 0x73, 0x89, 0x0c, 0x56,
				     0xe4, 0x31, 0xdf, 0x70, 0x3c, 0x40,
				     0x0d, 0x36, 0xfc, 0x4a, 0xf3, 0xa2,
				     0x8f, 0x9a, 0x9d, 0xaa, 0xb0 ),
			    BIGINT ( 0xbc, 0xca, 0x45 ),
			    BIGINT ( 0x9f, 0x5f, 0x7c, 0xac, 0x5e, 0xc7,
				     0xf2, 0xc5, 0x72, 0x3d, 0xff, 0x29,
				     0xd2, 0x25, 0xa9, 0x64, 0x5b, 0xbe,
				     0x63, 0x63, 0xc6, 0x84, 0x20 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xf8, 0xc9, 0xb9, 0x3d, 0xe1, 0xff,
				     0xa6, 0x8e, 0xb0, 0xd2, 0xa9, 0xa9,
				     0xc1, 0x5c, 0xc5, 0x94, 0x90, 0xb9,
				     0xca, 0x2f, 0x1a, 0xbd, 0x21 ),
			    BIGINT ( 0xa7, 0xf4, 0xb0, 0x3c, 0xf4, 0x2b,
				     0x9d, 0x40, 0x5f, 0xfd, 0x2e, 0x28,
				     0xa9, 0x23, 0x01, 0xaf, 0x0b, 0x73,
				     0xaa, 0xcf, 0x14, 0xdc, 0xd8 ),
			    BIGINT ( 0x31, 0xe2, 0xe8, 0xf0 ),
			    BIGINT ( 0x53, 0x30, 0xc6, 0x10, 0x12, 0x7c,
				     0xb3, 0x91, 0x15, 0x5f, 0x01, 0x62,
				     0xec, 0x1f, 0x15, 0x61, 0x3b, 0x9a,
				     0x76, 0x22, 0xf8, 0x31, 0xb1 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xff, 0x8c, 0x04, 0x74, 0x3e, 0x93,
				     0xfd, 0xce, 0xd5, 0x7f, 0xc5, 0x58,
				     0xce, 0x00, 0x53, 0x44, 0x02, 0xf4,
				     0xfd, 0x01, 0xc3, 0xb0, 0x3c ),
			    BIGINT ( 0x2f, 0xbe, 0xb3, 0x2d, 0xd6, 0x59,
				     0x69, 0x44, 0xc0, 0xd4, 0x27, 0x9c,
				     0xff, 0x53, 0x9e, 0x66, 0x2c, 0x01,
				     0x3a, 0x96, 0x5d, 0x75, 0xc1 ),
			    BIGINT ( 0x47, 0x3e, 0xb2, 0x81, 0x51, 0x9a,
				     0xdf, 0x75, 0xba, 0xa5, 0x19, 0xc1,
				     0xc7, 0xcc, 0xae, 0x82, 0x9c, 0x3e,
				     0xfd, 0x7f, 0xb0, 0xd7, 0x00 ),
			    BIGINT ( 0x09, 0x9c, 0xd0, 0x49, 0x1d, 0x88,
				     0xd8, 0x08, 0x45, 0x61, 0x71, 0xa1,
				     0xb5, 0xab, 0xa9, 0x5b, 0xa8, 0xf1,
				     0xc6, 0x53, 0x68, 0x8f, 0x3e ) );
	bigint_mod_exp_ok ( BIGINT ( 0xd8, 0x78, 0xad, 0x80, 0x81, 0xf1,
				     0x84, 0x23, 0x82, 0x5d, 0x49, 0x46,
				     0x75, 0xfd, 0xd1, 0x49, 0x53, 0x10,
				     0x4d, 0x10, 0xab, 0x0f, 0xf0 ),
			    BIGINT ( 0x78, 0x3d, 0x09, 0x1b, 0xea, 0xa4,
				     0xb9, 0x13, 0xf8, 0xb5, 0xb5, 0x5e,
				     0x69, 0xa4, 0xe1, 0xfd, 0x88, 0x58,
				     0x26, 0xb3, 0x76, 0xa2, 0x38 ),
			    BIGINT ( 0x3b, 0x12, 0xe0, 0x8e, 0xa2, 0x2f,
				     0x2a, 0x2b, 0xb1, 0x78, 0xf9, 0xf6,
				     0x93, 0x4d, 0x52, 0x82, 0x29, 0x2d,
				     0xe4, 0x36, 0x92, 0x49, 0xc1, 0x25,
				     0x6e, 0x26, 0xe6, 0x6e, 0xc2, 0x4d,
				     0xea, 0x13, 0x86, 0x85, 0x71, 0x4d,
				     0x85, 0x70, 0xf9, 0x2b, 0xa0, 0x0f,
				     0x96, 0xe5, 0x63, 0x7a, 0xb4, 0x25,
				     0x53, 0x1a, 0xd8, 0x30, 0x36, 0xba,
				     0x6e, 0x2e, 0xce, 0x2d, 0x8f, 0x32,
				     0xe9, 0xdc, 0x91, 0x9e, 0xd4, 0xf1,
				     0x3b, 0x40, 0xc9, 0xf4, 0x97, 0x74,
				     0x5e, 0x69, 0xcd, 0x34, 0x4a, 0x18,
				     0x65, 0xe5, 0x07, 0xb5, 0x9e, 0x2a,
				     0xc4, 0xeb, 0xb6, 0x96, 0x7b, 0x99,
				     0x0c, 0xe4, 0xb3, 0x85, 0xff, 0x17,
				     0x72, 0x5d, 0xf6, 0x30, 0xb4, 0xff,
				     0x98, 0xe6, 0xf6, 0x31, 0x24, 0x82,
				     0x91, 0xa6, 0x18, 0x6d, 0x0b, 0x84,
				     0x6f, 0x5f, 0x64, 0xa3, 0xdf, 0x92,
				     0x06, 0x16, 0xe3, 0x7c, 0x08, 0x61,
				     0x77, 0xce ),
			    BIGINT ( 0x17, 0xc9, 0xc5, 0x38, 0x4c, 0x15,
				     0x0f, 0x4e, 0xc2, 0x90, 0x3b, 0x46,
				     0x7b, 0x2f, 0x95, 0x82, 0xfe, 0x51,
				     0x95, 0x2b, 0xff, 0xd5, 0x28 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x69, 0xa3, 0x7e, 0x24, 0xdf, 0x9e,
				     0x0b, 0x3e, 0x3f, 0x43, 0x06, 0x0e,
				     0x1d, 0x57, 0x74, 0xe0, 0xa0, 0x5b,
				     0x82, 0xca, 0xb0, 0x33, 0x8b, 0xe4,
				     0x39, 0x27, 0x41, 0xd4, 0x2e, 0x30,
				     0x3a, 0x0e, 0x62, 0x6f, 0xfa, 0xb4,
				     0x02, 0x88, 0x70, 0x35, 0xa6, 0xea,
				     0x7d, 0xb2, 0x87, 0xc3, 0xa5, 0x50,
				     0x49, 0x38, 0xa4, 0x68, 0xa9, 0xe4,
				     0xa6, 0xcc, 0xd7, 0x13, 0xb1, 0xd9,
				     0x1c, 0x6a, 0x9a, 0xb8, 0x6c, 0x9b,
				     0xff, 0xcd, 0x2c, 0xb3, 0xbd, 0xe2,
				     0xfd, 0x1f, 0x08, 0xdd, 0xc6, 0xee,
				     0x18, 0x0c, 0xa5, 0xcd, 0x09, 0x19,
				     0x51, 0x51, 0xa5, 0x6f, 0x93, 0x1b,
				     0x34, 0xfd, 0x8f, 0xd9, 0x87, 0xed,
				     0x15, 0x7e, 0x36, 0x60, 0xdd, 0x1b,
				     0xf4, 0xcc, 0xc4, 0x4c, 0x19, 0x2b,
				     0xd6, 0x1e, 0xec, 0x51, 0xe9, 0x27,
				     0xe9, 0xbd, 0x6a, 0x3f, 0x91, 0x45,
				     0xc3, 0x6d, 0x40, 0x7e, 0x6c, 0x56,
				     0x05, 0x5a ),
			    BIGINT ( 0x5c, 0x96, 0x05, 0x81, 0x94, 0x45,
				     0xcf, 0x47, 0x5f, 0x1b, 0xb0, 0xf9,
				     0xef, 0x13, 0x8f, 0xcc, 0x71, 0xfd,
				     0x50, 0xf1, 0xe7, 0x62, 0x6e, 0xfa,
				     0x48, 0x66, 0x1c, 0xf7, 0xef, 0x09,
				     0x12, 0xa2, 0xfd, 0x17, 0xb7, 0x6a,
				     0x3b, 0xed, 0xf7, 0x86, 0xd2, 0xbe,
				     0x95, 0x90, 0xc6, 0x00, 0x14, 0x8d,
				     0xe3, 0x27, 0xbe, 0x03, 0x7c, 0x9e,
				     0x6b, 0x51, 0x31, 0x8d, 0x18, 0xc4,
				     0x16, 0xd2, 0x84, 0x63, 0x9b, 0xe9,
				     0xa4, 0xf8, 0xff, 0x70, 0x4d, 0xeb,
				     0x6f, 0x4a, 0xb7, 0x5b, 0x54, 0xf1,
				     0xb5, 0xbe, 0x78, 0xb6, 0xfd, 0x8b,
				     0xe1, 0x39, 0x62, 0x85, 0x9b, 0xde,
				     0x30, 0xa8, 0xe4, 0x37, 0x52, 0x57,
				     0x39, 0x79, 0xdb, 0x0b, 0x19, 0x6b,
				     0xc9, 0x17, 0xfd, 0x8c, 0x2c, 0xaa,
				     0xa4, 0xf1, 0x04, 0xd1, 0xd3, 0x2f,
				     0xbb, 0x3a, 0x36, 0x82, 0x31, 0xa4,
				     0x40, 0xd4, 0x87, 0x46, 0xe3, 0x6e,
				     0xd0, 0x17 ),
			    BIGINT ( 0x93 ),
			    BIGINT ( 0x0d, 0x39, 0x92, 0x57, 0xaa, 0x6d,
				     0xfc, 0x3b, 0x10, 0x18, 0x6d, 0x59,
				     0xbe, 0x31, 0x8f, 0xee, 0xf9, 0x82,
				     0x84, 0xe0, 0xdf, 0xa5, 0x00, 0x28,
				     0xd1, 0x64, 0x6b, 0x4b, 0x43, 0x3b,
				     0x76, 0x3e, 0x6b, 0xc4, 0xe4, 0xf5,
				     0x0b, 0x59, 0x5a, 0xe4, 0x53, 0x5e,
				     0x02, 0xd4, 0xde, 0x72, 0xd3, 0xa3,
				     0x58, 0x66, 0xa7, 0xdd, 0x2b, 0x0b,
				     0xa4, 0x83, 0xd0, 0xd9, 0xef, 0x29,
				     0x3d, 0x2f, 0x97, 0xff, 0x9a, 0xc7,
				     0xf6, 0x8a, 0x8d, 0x59, 0xef, 0x87,
				     0xd1, 0xe6, 0xba, 0x4d, 0x99, 0xd9,
				     0x5f, 0x5e, 0x7a, 0x7e, 0x67, 0x22,
				     0x5b, 0x77, 0x83, 0xa2, 0x02, 0xfd,
				     0xb2, 0xe4, 0xf6, 0x20, 0x4c, 0x12,
				     0x20, 0xa7, 0xda, 0x5b, 0x3b, 0x8c,
				     0xa2, 0xca, 0xda, 0x20, 0xaa, 0x27,
				     0xe6, 0x54, 0x3e, 0xa8, 0x6f, 0x64,
				     0x9d, 0xa7, 0x0d, 0x57, 0x1b, 0x21,
				     0xff, 0xd2, 0xe2, 0xb2, 0x0a, 0x4f,
				     0xb7, 0x0e ) );
	bigint_mod_exp_ok ( BIGINT ( 0x06, 0xcf, 0x54, 0xf2, 0x0d, 0x62,
				     0x33, 0xdd, 0xe7, 0x4d, 0x7f, 0x2f,
				     0x8e, 0x52, 0x73, 0xf4, 0x73, 0x68,
				     0x4b, 0x13, 0x6e, 0x58, 0x6b, 0x4a,
				     0xb8, 0x4c, 0xef, 0x73, 0xfe, 0x5f,
				     0xf6, 0xd0, 0xbb, 0x82, 0x17, 0x3f,
				     0x9d, 0x91, 0xf8, 0xa3, 0xb8, 0x79,
				     0xef, 0x41, 0x38, 0xc1, 0xef, 0xc9,
				     0xc6, 0xcf, 0x2a, 0xc3, 0xaa, 0x75,
				     0x17, 0xda, 0xbc, 0x76, 0x29, 0x61,
				     0x6d, 0x05, 0x79, 0x0b, 0x44, 0xb1,
				     0x54, 0x75, 0xb7, 0xd9, 0xf6, 0xa8,
				     0xbd, 0xf7, 0x85, 0xe0, 0xe7, 0x90,
				     0x62, 0xce, 0x79, 0xfb, 0xc5, 0x23,
				     0xa5, 0x09, 0xc0, 0xc4, 0x4d, 0xe7,
				     0x9c, 0x49, 0x8f, 0x82, 0xf1, 0x31,
				     0x34, 0x85, 0xdd, 0x3b, 0xbe, 0xe9,
				     0x93, 0x19, 0x03, 0x75, 0x3f, 0xc4,
				     0xa4, 0x0f, 0x52, 0x53, 0xc1, 0xcd,
				     0x08, 0xb0, 0x05, 0x0c, 0xa2, 0x0c,
				     0x3a, 0x72, 0xb2, 0x3c, 0xdb, 0x4f,
				     0xac, 0xc6 ),
			    BIGINT ( 0xe4, 0x40, 0xd8, 0x30, 0x00, 0xcf,
				     0x4c, 0xfd, 0xda, 0xae, 0x90, 0xd3,
				     0x5b, 0xc7, 0x20, 0xcc, 0x2b, 0xe2,
				     0x0a, 0x39, 0x1e, 0xde, 0xef, 0x98,
				     0x16, 0x3b, 0x9d, 0x36, 0x63, 0x0d,
				     0x46, 0xed, 0x23, 0x6e, 0x38, 0xa8,
				     0x15, 0xb5, 0xb1, 0xaf, 0x47, 0xb1,
				     0xec, 0xaa, 0x8b, 0x57, 0xd6, 0xca,
				     0x39, 0x2f, 0x62, 0xbd, 0xd5, 0xf8,
				     0x98, 0x98, 0x5d, 0xfe, 0x14, 0xd6,
				     0xdc, 0xe5, 0x98, 0x60, 0x5b, 0x16,
				     0x92, 0xcb, 0xed, 0xb6, 0x9c, 0x5c,
				     0x82, 0x40, 0x6b, 0xaa, 0x48, 0x7a,
				     0xd4, 0xfe, 0xa3, 0xe7, 0x30, 0xf1,
				     0x7c, 0xfb, 0x94, 0x2e, 0xeb, 0xb6,
				     0x71, 0xe4, 0x33, 0x63, 0xc3, 0xb0,
				     0x94, 0x6d, 0xee, 0xa5, 0x15, 0x3f,
				     0x28, 0xf1, 0xfa, 0xdc, 0xf2, 0x13,
				     0x0f, 0xc7, 0xd9, 0xe0, 0xbf, 0x1b,
				     0x49, 0xee, 0x21, 0x8e, 0x26, 0xc9,
				     0x28, 0x21, 0x86, 0x1d, 0x46, 0x33,
				     0xd4, 0x69 ),
			    BIGINT ( 0xd9, 0x87 ),
			    BIGINT ( 0xdf, 0xff, 0xcc, 0xb7, 0xfe, 0x19,
				     0x02, 0x92, 0x9d, 0xab, 0x33, 0xd2,
				     0x21, 0xbc, 0xd3, 0xc4, 0x31, 0xad,
				     0x4b, 0xb3, 0x16, 0x50, 0x96, 0xd9,
				     0xdc, 0x88, 0x74, 0x60, 0xde, 0xdf,
				     0xb7, 0x83, 0xdb, 0x22, 0xef, 0xcb,
				     0xcb, 0xdb, 0x4c, 0xfb, 0x94, 0x4c,
				     0x3f, 0xf5, 0xf5, 0x99, 0x85, 0x21,
				     0x1a, 0x2b, 0xec, 0x90, 0x2d, 0xb4,
				     0x20, 0x3c, 0x27, 0x9f, 0xe5, 0xb1,
				     0x5c, 0x92, 0xfa, 0xb0, 0xa9, 0x8e,
				     0x2c, 0x21, 0x8e, 0x8d, 0xe5, 0x55,
				     0x84, 0x02, 0xa5, 0x15, 0x5c, 0x53,
				     0x1f, 0x40, 0x81, 0x0a, 0x10, 0xde,
				     0x21, 0x41, 0xa9, 0x97, 0xf8, 0x6f,
				     0xbf, 0x42, 0x58, 0x9e, 0xc6, 0xdd,
				     0x10, 0x33, 0x3f, 0xad, 0xe6, 0x8e,
				     0x57, 0x27, 0x37, 0x20, 0xa4, 0x86,
				     0xef, 0x39, 0x7b, 0x6f, 0x78, 0x77,
				     0xab, 0xa0, 0x62, 0xe1, 0xfd, 0x9c,
				     0xbe, 0xfa, 0x98, 0x2e, 0x29, 0xe3,
				     0xeb, 0x52 ) );
	bigint_mod_exp_ok ( BIGINT ( 0x00, 0x91, 0xb3, 0x87, 0xe6, 0x01,
				     0x57, 0xe9, 0x68, 0xa4, 0xf4, 0x9b,
				     0xea, 0x6a, 0x8a, 0x9e, 0x1a, 0x8b,
				     0xd3, 0x85, 0x9d, 0xba, 0x85, 0xab,
				     0xd8, 0xcd, 0x25, 0x56, 0x8e, 0x85,
				     0x8a, 0x8e, 0x48, 0x9e, 0xb4, 0x90,
				     0xc8, 0x2e, 0x07, 0x78, 0x80, 0x49,
				     0xa0, 0xb7, 0x95, 0x6a, 0xd8, 0xad,
				     0xb5, 0xda, 0x5d, 0xe6, 0x11, 0x87,
				     0xb8, 0x33, 0x8f, 0xa8, 0x6f, 0x4e,
				     0xc6, 0xc3, 0x0d, 0xf5, 0xa9, 0x4e,
				     0xb2, 0x42, 0x53, 0x81, 0xcd, 0x33,
				     0x83, 0x49, 0xab, 0x0d, 0x0e, 0xf5,
				     0x2c, 0xcd, 0x84, 0x58, 0xf3, 0x30,
				     0xa3, 0x6e, 0x3c, 0x3a, 0xc6, 0x77,
				     0x43, 0xb0, 0xe7, 0x4b, 0x66, 0x30,
				     0xe9, 0x48, 0x0b, 0x0d, 0x86, 0x3f,
				     0xd8, 0xe2, 0xb5, 0x88, 0xc1, 0x44,
				     0xb2, 0x6b, 0xb0, 0x7a, 0x35, 0x3b,
				     0x56, 0x83, 0xb1, 0xac, 0x9e, 0xeb,
				     0x9b, 0x08, 0x43, 0xac, 0x0a, 0x3a,
				     0x31, 0x69 ),
			    BIGINT ( 0x96, 0x6f, 0xb0, 0xa7, 0x02, 0xb5,
				     0xd9, 0x19, 0xbe, 0x4b, 0x27, 0x65,
				     0x5b, 0x96, 0xd4, 0x0b, 0x49, 0x70,
				     0xf0, 0x09, 0x8e, 0xf2, 0x04, 0x85,
				     0x93, 0xe9, 0x2e, 0x09, 0x31, 0x76,
				     0x8b, 0xbb, 0xe9, 0xe1, 0x2b, 0x4f,
				     0xed, 0x83, 0xa6, 0x87, 0xa3, 0x07,
				     0x0a, 0x3d, 0x1c, 0x65, 0x14, 0x5a,
				     0xd5, 0xc0, 0x5d, 0x3c, 0x31, 0x9a,
				     0x83, 0xad, 0xca, 0x6a, 0x93, 0x0d,
				     0x1a, 0x67, 0x4e, 0x68, 0x06, 0x64,
				     0x53, 0x2e, 0x15, 0xd9, 0xdd, 0x5e,
				     0xcb, 0xb7, 0x2e, 0xef, 0xd3, 0xbb,
				     0x5f, 0xaf, 0xef, 0x9e, 0xf2, 0x7b,
				     0x69, 0x15, 0xb0, 0x18, 0x6c, 0x67,
				     0x10, 0xda, 0x33, 0x07, 0x48, 0x97,
				     0x31, 0xb3, 0x3d, 0x3d, 0xc9, 0x2e,
				     0x0b, 0x68, 0x91, 0x3f, 0x6a, 0x3b,
				     0x1a, 0xdf, 0xa8, 0x69, 0x46, 0x1c,
				     0xb2, 0x69, 0x08, 0x0b, 0x02, 0x1b,
				     0x03, 0x64, 0xae, 0xb6, 0x2d, 0xc6,
				     0xc4, 0x0a ),
			    BIGINT ( 0x6d, 0x3f, 0xdd ),
			    BIGINT ( 0x40, 0x6e, 0x9d, 0x3e, 0xeb, 0xa4,
				     0xb1, 0x8d, 0xb7, 0xb4, 0x0f, 0x5b,
				     0x12, 0xad, 0x27, 0x9e, 0xbd, 0xe7,
				     0xe5, 0x9d, 0xec, 0xb4, 0xac, 0x23,
				     0x5f, 0xa9, 0xec, 0x9c, 0xd1, 0x6a,
				     0xbe, 0x99, 0xba, 0xb3, 0x66, 0x0e,
				     0x17, 0xaa, 0x13, 0xa2, 0x2e, 0x01,
				     0x28, 0xb1, 0x6c, 0xba, 0xad, 0x68,
				     0x48, 0xf0, 0xf3, 0x4c, 0x08, 0x9f,
				     0xd1, 0x9c, 0xb7, 0x75, 0xc5, 0xb6,
				     0x5a, 0x05, 0xb0, 0x14, 0xd4, 0x61,
				     0xea, 0x18, 0x9f, 0xe6, 0xe5, 0xe3,
				     0xd4, 0xff, 0x35, 0x43, 0x0b, 0xb8,
				     0xf6, 0xe9, 0x19, 0x7a, 0x88, 0xa7,
				     0x4d, 0x01, 0x92, 0x05, 0xd2, 0x6e,
				     0xa3, 0xc1, 0xb6, 0x66, 0x75, 0xb1,
				     0x00, 0x0d, 0x42, 0x37, 0xcc, 0xca,
				     0xc0, 0x8d, 0xc8, 0x7e, 0x5c, 0xc9,
				     0x53, 0x81, 0x2f, 0xc4, 0x61, 0xb6,
				     0x96, 0x3b, 0xa5, 0x04, 0x14, 0x1b,
				     0xa7, 0x77, 0xa1, 0xbc, 0x73, 0x1d,
				     0xad, 0xed ) );
	bigint_mod_exp_ok ( BIGINT ( 0x45, 0xfb, 0xf3, 0xdc, 0x31, 0xe5,
				     0x56, 0x7a, 0xee, 0x15, 0xfb, 0x16,
				     0xee, 0x6e, 0x90, 0x3e, 0xa3, 0x89,
				     0xc2, 0x6d, 0x9b, 0x06, 0x65, 0xd0,
				     0xcd, 0xa2, 0xcc, 0x01, 0x60, 0x0d,
				     0xd1, 0xdd, 0x68, 0x14, 0xc2, 0xcd,
				     0xd8, 0x79, 0x75, 0xad, 0x0a, 0x9f,
				     0x39, 0x5f, 0x52, 0x4b, 0x58, 0x31,
				     0x48, 0xbb, 0x2a, 0xcc, 0xe0, 0x42,
				     0x18, 0x32, 0xdc, 0x63, 0x14, 0x11,
				     0x4e, 0xab, 0x96, 0x29, 0xc5, 0x06,
				     0x79, 0xe5, 0x06, 0xf7, 0x59, 0xdb,
				     0x1e, 0x51, 0xfd, 0xc4, 0x48, 0x3a,
				     0x4c, 0x7f, 0xd0, 0xe2, 0x36, 0x86,
				     0xc1, 0x8b, 0xc5, 0x86, 0x52, 0xe0,
				     0xdb, 0x92, 0x5f, 0x0e, 0x19, 0xb1,
				     0xa3, 0x23, 0xdd, 0xf0, 0x78, 0xcc,
				     0x81, 0x3f, 0x4a, 0xe6, 0xb0, 0x32,
				     0xd1, 0x5c, 0x5e, 0x3a, 0xb0, 0xd8,
				     0xe2, 0x04, 0xc0, 0x30, 0x85, 0x1d,
				     0x5e, 0x28, 0xee, 0xd9, 0xb3, 0x83,
				     0x9f, 0xe2 ),
			    BIGINT ( 0xb3, 0x2c, 0x2e, 0xc5, 0xba, 0xf8,
				     0x41, 0x98, 0x79, 0x7e, 0xaa, 0x0c,
				     0x2a, 0x8f, 0xd9, 0x56, 0x55, 0xaa,
				     0x74, 0x60, 0x74, 0xd1, 0x49, 0x2c,
				     0x6f, 0x0a, 0x4e, 0xf8, 0x3f, 0x1b,
				     0x73, 0x4c, 0xe0, 0x17, 0x37, 0x06,
				     0x76, 0x73, 0xd5, 0x2d, 0x4d, 0x3f,
				     0xb0, 0x15, 0x7e, 0x98, 0xd0, 0xdf,
				     0xf0, 0x33, 0x78, 0xe2, 0xe6, 0xec,
				     0x21, 0x22, 0xad, 0xd5, 0xab, 0x2d,
				     0x0d, 0x59, 0x95, 0x05, 0x34, 0x1f,
				     0x51, 0xf5, 0xec, 0x93, 0x05, 0x15,
				     0x37, 0xcf, 0x93, 0x03, 0xd7, 0xf6,
				     0x35, 0x23, 0x8f, 0x33, 0xf6, 0xba,
				     0x42, 0xc8, 0x52, 0x94, 0xd3, 0x33,
				     0x3e, 0x39, 0x01, 0xd1, 0x55, 0x3f,
				     0x48, 0x84, 0xe9, 0xbc, 0x0b, 0x0f,
				     0xc9, 0x69, 0x41, 0x2c, 0x5f, 0x34,
				     0xd0, 0xe6, 0x15, 0x50, 0x06, 0x64,
				     0x5b, 0x8b, 0x71, 0x22, 0xb3, 0x3e,
				     0x09, 0x9c, 0x76, 0x13, 0x9b, 0x29,
				     0x57, 0x94 ),
			    BIGINT ( 0xca, 0x94, 0xf7, 0xca ),
			    BIGINT ( 0x83, 0x68, 0xb9, 0xe7, 0x91, 0xf3,
				     0x3b, 0x5a, 0x0b, 0xb6, 0x1e, 0x2f,
				     0x3f, 0x5f, 0xdc, 0x96, 0x5b, 0x7f,
				     0x8d, 0xc5, 0x8e, 0xda, 0x6e, 0x21,
				     0xe3, 0x20, 0xea, 0x37, 0x39, 0x3b,
				     0xb4, 0xd7, 0xf6, 0xba, 0x61, 0xfe,
				     0xdc, 0x7e, 0x82, 0x9a, 0x38, 0x7b,
				     0xd5, 0xb1, 0x11, 0x98, 0xc4, 0x88,
				     0x0b, 0x01, 0x7d, 0x81, 0xc9, 0x64,
				     0x23, 0xc3, 0x3e, 0xf3, 0x67, 0x95,
				     0x78, 0xca, 0xda, 0x52, 0xaf, 0x72,
				     0x25, 0xd9, 0xf0, 0x27, 0xd3, 0x1c,
				     0xfb, 0xad, 0xa1, 0xa7, 0x06, 0x2f,
				     0xaa, 0x2f, 0x86, 0x5c, 0x8b, 0x30,
				     0xe1, 0xda, 0x5a, 0x36, 0xf9, 0xfd,
				     0xbf, 0xfe, 0x0d, 0x03, 0xf8, 0x9c,
				     0x6b, 0x9b, 0xe5, 0x70, 0x6d, 0x75,
				     0xd7, 0x54, 0x28, 0x43, 0x34, 0x69,
				     0x98, 0x11, 0x29, 0xee, 0x50, 0x06,
				     0xa4, 0xc4, 0x11, 0x6d, 0x60, 0x8c,
				     0xcd, 0xd1, 0x88, 0xe9, 0x6b, 0xbb,
				     0xc1, 0xd4 ) );
	bigint_mod_exp_ok ( BIGINT ( 0xa1, 0x01, 0x7e, 0xb4, 0x0e, 0x66,
				     0xa5, 0x07, 0x8b, 0x10, 0x84, 0x0d,
				     0x30, 0x0a, 0xa4, 0x2d, 0x10, 0x2c,
				     0xd4, 0x9a, 0x27, 0xf1, 0x02, 0x8c,
				     0x38, 0x18, 0x7f, 0x7f, 0x95, 0x65,
				     0xf1, 0xa9, 0x3b, 0x7d, 0x1f, 0x4f,
				     0x88, 0xb0, 0x65, 0x62, 0x63, 0x63,
				     0xaa, 0x82, 0xfc, 0x83, 0x3a, 0x3a,
				     0x46, 0x59, 0x6a, 0x89, 0xec, 0xa9,
				     0xb0, 0x4c, 0x5e, 0xbe, 0x46, 0x98,
				     0xd0, 0xd4, 0xb7, 0xe3, 0x1b, 0x30,
				     0x0b, 0xfb, 0xbb, 0x4f, 0x0b, 0xd3,
				     0xe4, 0xa0, 0x80, 0x54, 0xcb, 0x52,
				     0x0a, 0xe8, 0x03, 0x75, 0x8e, 0x96,
				     0xa4, 0x21, 0xaa, 0xbd, 0x7a, 0xfd,
				     0xfa, 0xf8, 0xaf, 0x42, 0xf6, 0x61,
				     0xd2, 0x93, 0xce, 0x66, 0x67, 0xe9,
				     0x02, 0xda, 0x81, 0x0b, 0xb0, 0x1e,
				     0x9e, 0x27, 0x57, 0x98, 0x18, 0x88,
				     0x35, 0x49, 0xc0, 0x88, 0x88, 0x59,
				     0xae, 0x2f, 0x66, 0x59, 0x31, 0x87,
				     0x88, 0xda ),
			    BIGINT ( 0xfe, 0x21, 0x7c, 0xf4, 0xbe, 0xae,
				     0x65, 0xda, 0x89, 0xd2, 0x26, 0xd6,
				     0x9c, 0x65, 0xc6, 0xb6, 0xb4, 0x0a,
				     0x84, 0x11, 0xe1, 0xe8, 0xba, 0xd8,
				     0x16, 0xcf, 0x60, 0x6c, 0x83, 0xa5,
				     0x4a, 0xbf, 0xa2, 0x24, 0x0b, 0x66,
				     0xda, 0xe2, 0x4e, 0x2d, 0xe5, 0x9e,
				     0xbf, 0xad, 0x5c, 0xa3, 0x1e, 0x5c,
				     0xbd, 0xe2, 0x5b, 0x46, 0xcf, 0xcc,
				     0xd5, 0xc9, 0x13, 0x95, 0xc3, 0xdb,
				     0x64, 0xbf, 0xeb, 0x31, 0xa9, 0x8a,
				     0x3b, 0xd2, 0x5d, 0x3b, 0x2e, 0xdc,
				     0x0c, 0xca, 0xab, 0xde, 0x92, 0xae,
				     0x45, 0x35, 0x96, 0xb0, 0xb7, 0xb9,
				     0xe6, 0xfe, 0x28, 0x0d, 0x10, 0x72,
				     0x53, 0x8e, 0x21, 0xc0, 0x33, 0x79,
				     0x01, 0x43, 0x8d, 0x77, 0xc4, 0xaa,
				     0xcf, 0x7f, 0xc3, 0xd1, 0xf5, 0xfd,
				     0x79, 0x81, 0xf6, 0x2e, 0xb7, 0xeb,
				     0x55, 0x5f, 0x74, 0xf0, 0x3a, 0xb9,
				     0x57, 0x07, 0x09, 0x97, 0xa5, 0x4c,
				     0x4a, 0x85 ),
			    BIGINT ( 0xd9, 0xb7, 0xb2, 0xd6, 0xeb, 0xf3,
				     0x66, 0xbe, 0x15, 0x64, 0xad, 0x2e,
				     0x9e, 0xc6, 0xaf, 0x5e, 0xaf, 0x40,
				     0x1e, 0x90, 0x82, 0x2f, 0x98 ),
			    BIGINT ( 0x12, 0x48, 0x31, 0x7f, 0x09, 0xbb,
				     0x8f, 0xd9, 0x02, 0x7e, 0x4a, 0xd0,
				     0x2f, 0x42, 0x7c, 0x17, 0x6e, 0x83,
				     0x74, 0x21, 0x95, 0x47, 0x7d, 0x93,
				     0x4a, 0xce, 0x34, 0x7c, 0xde, 0xc7,
				     0x8f, 0xf6, 0x28, 0x97, 0xba, 0x81,
				     0x9b, 0xcc, 0x54, 0x14, 0x7f, 0xd3,
				     0x93, 0x66, 0x41, 0x8c, 0x0e, 0x47,
				     0xee, 0xc5, 0x5e, 0xd6, 0x5f, 0x01,
				     0x62, 0x97, 0xf1, 0x2b, 0xee, 0x60,
				     0x5e, 0x82, 0x2c, 0x7b, 0x0a, 0xf2,
				     0xc3, 0x23, 0xbf, 0xb9, 0x83, 0xf7,
				     0x97, 0xf5, 0xca, 0x58, 0xd7, 0xf0,
				     0x87, 0x7b, 0xcb, 0x87, 0x69, 0x42,
				     0xbc, 0x05, 0xc4, 0xad, 0xbd, 0x82,
				     0xcf, 0x44, 0x16, 0x4f, 0x46, 0xe0,
				     0xde, 0x2f, 0xfa, 0x77, 0xec, 0xa4,
				     0x23, 0x7d, 0x47, 0x3e, 0x94, 0x19,
				     0x8b, 0xb8, 0x84, 0x81, 0x80, 0x6c,
				     0x1e, 0x31, 0xa3, 0x6d, 0x14, 0x94,
				     0x57, 0x28, 0x99, 0x08, 0x0a, 0xa7,
				     0x98, 0x4b ) );
	bigint_mod_exp_ok ( BIGINT ( 0xda, 0x52, 0xfd, 0x44, 0x5d, 0x11,
				     0x60, 0x6c, 0xec, 0x87, 0xbf, 0x19,
				     0xb8, 0x46, 0xaa, 0x41, 0xfc, 0x10,
				     0xae, 0x47, 0xd6, 0x72, 0x42, 0x57,
				     0xc3, 0x05, 0xca, 0xe3, 0x59, 0x94,
				     0x82, 0x7c, 0xa1, 0xe0, 0xd2, 0x6b,
				     0x77, 0x71, 0x42, 0xa1, 0xf7, 0x84,
				     0xae, 0xf4, 0x6f, 0x44, 0x0d, 0x88,
				     0xa2, 0xc5, 0x45, 0x9b, 0x49, 0x36,
				     0xd4, 0x20, 0x3a, 0x7c, 0x92, 0xdb,
				     0x65, 0xd9, 0x20, 0xd6, 0x71, 0x22,
				     0x90, 0x70, 0xbf, 0xf3, 0x17, 0xe8,
				     0x2c, 0x10, 0xe9, 0x4c, 0x02, 0x69,
				     0x37, 0xa2, 0x91, 0x04, 0x46, 0x11,
				     0xdc, 0xab, 0x5b, 0x1e, 0x3e, 0x31,
				     0xd8, 0x69, 0xf8, 0x48, 0x84, 0x1f,
				     0x56, 0x46, 0xf1, 0xc0, 0x14, 0x3f,
				     0xcc, 0x5d, 0xe2, 0xf7, 0x8b, 0xa4,
				     0x9e, 0x94, 0x32, 0xaa, 0x3c, 0x5e,
				     0x21, 0x70, 0x00, 0x24, 0x2a, 0x1b,
				     0xec, 0x25, 0xb1, 0xb6, 0x83, 0x36,
				     0x5a, 0x95 ),
			    BIGINT ( 0x5e, 0xdc, 0x71, 0x1f, 0x5b, 0x55,
				     0xaa, 0xda, 0x56, 0xf5, 0x93, 0x9b,
				     0xe8, 0xfc, 0x6a, 0x80, 0xe1, 0xe3,
				     0x93, 0xe4, 0xc0, 0x58, 0x6f, 0x22,
				     0xce, 0x9d, 0x6f, 0x84, 0x4c, 0xd4,
				     0x12, 0x44, 0x57, 0x25, 0xca, 0xe5,
				     0x2b, 0x7c, 0x35, 0x88, 0xc7, 0x38,
				     0x25, 0x20, 0x9b, 0x57, 0xf2, 0xf2,
				     0x6c, 0x28, 0x47, 0x9c, 0x3f, 0x91,
				     0x1e, 0x3f, 0xe9, 0xeb, 0x50, 0xd6,
				     0xa7, 0x22, 0x88, 0x6c, 0x71, 0xe5,
				     0x62, 0x2a, 0xb7, 0xce, 0xbe, 0xf7,
				     0x1a, 0x8c, 0x52, 0xa6, 0xff, 0xb8,
				     0x34, 0x83, 0x7e, 0x04, 0xa8, 0x9c,
				     0xa8, 0xa7, 0xd1, 0x05, 0x8e, 0x13,
				     0x03, 0xe0, 0x49, 0xd8, 0x4a, 0xc4,
				     0x4d, 0x38, 0x21, 0x5b, 0x62, 0xc2,
				     0x38, 0x23, 0x7c, 0x9e, 0xf1, 0xe9,
				     0xb6, 0x9a, 0x75, 0x42, 0x14, 0x99,
				     0x63, 0x36, 0x13, 0x4c, 0x2d, 0x3a,
				     0x77, 0xd4, 0x74, 0xb7, 0x30, 0xb2,
				     0x00, 0x0f ),
			    BIGINT ( 0xe3, 0xe5, 0x3b, 0xb5, 0x92, 0x5a,
				     0xc6, 0xfa, 0x8f, 0xe8, 0x00, 0xb9,
				     0x5c, 0xa0, 0xb6, 0x3e, 0x5e, 0x14,
				     0x12, 0xa9, 0xdd, 0x2a, 0x3d, 0x4d,
				     0xa3, 0x91, 0x6a, 0x56, 0x99, 0xc2,
				     0x6c, 0x8e, 0xda, 0xb0, 0x5a, 0x2a,
				     0x37, 0x55, 0x8b, 0xd3, 0x9b, 0xb6,
				     0x1d, 0x49, 0x7d, 0x81, 0x76, 0x1c,
				     0x2e, 0xb9, 0x92, 0x6d, 0xfa, 0x54,
				     0x53, 0xfc, 0x74, 0x9b, 0x6b, 0x63,
				     0x95, 0x1a, 0x89, 0xcc, 0xbd, 0x36,
				     0xc5, 0x31, 0x7f, 0xf5, 0x31, 0x69,
				     0x40, 0xd5, 0x7b, 0x94, 0x5d, 0xa9,
				     0xd1, 0x34, 0x95, 0xa1, 0x8b, 0xa5,
				     0xb5, 0x83, 0xda, 0xb5, 0x9d, 0x5b,
				     0x74, 0x41, 0xad, 0x81, 0x45, 0x40,
				     0x9b, 0xc3, 0xe8, 0xfe, 0x47, 0xdc,
				     0xb0, 0xc3, 0x34, 0x5d, 0xf6, 0x3c,
				     0x1d, 0x07, 0x76, 0xd9, 0x25, 0xca,
				     0xa2, 0x39, 0x6c, 0xa8, 0xae, 0x30,
				     0x4a, 0xde, 0xfb, 0xeb, 0x19, 0x80,
				     0x5e, 0x49 ),
			    BIGINT ( 0x4b, 0x0e, 0x74, 0xb8, 0xa7, 0x92,
				     0x74, 0xd9, 0x50, 0xf6, 0x1b, 0x67,
				     0x76, 0x76, 0x56, 0x6c, 0x09, 0x9c,
				     0x01, 0xda, 0xaf, 0xa3, 0xca, 0xb2,
				     0x12, 0x85, 0x52, 0x24, 0xe9, 0x7e,
				     0x2b, 0xf2, 0x6e, 0xe9, 0x1a, 0x10,
				     0x5d, 0xa0, 0x25, 0x46, 0x8f, 0x2a,
				     0x95, 0x62, 0x50, 0xb6, 0x66, 0x43,
				     0x37, 0x8b, 0xcb, 0x05, 0xf8, 0x61,
				     0x59, 0xf9, 0xdd, 0xd2, 0x68, 0x72,
				     0xfa, 0x88, 0x13, 0x36, 0xd8, 0x24,
				     0x73, 0xec, 0x47, 0x44, 0xdd, 0x45,
				     0x8a, 0x59, 0xd2, 0xbd, 0x43, 0xe3,
				     0x05, 0x16, 0xd5, 0x9b, 0x1c, 0x8a,
				     0x4b, 0x07, 0xda, 0x58, 0x0d, 0x4a,
				     0x4e, 0xe7, 0x15, 0xfc, 0xbd, 0x95,
				     0xf7, 0x18, 0xa5, 0xa7, 0x93, 0xff,
				     0xf8, 0x1f, 0xd4, 0x6b, 0x07, 0xc6,
				     0x5d, 0x90, 0x73, 0x57, 0x57, 0x37,
				     0xfa, 0x83, 0xd4, 0x7c, 0xe9, 0x77,
				     0x46, 0x91, 0x3a, 0x50, 0x0d, 0x6a,
				     0x25, 0xd0 ) );
}

/** Big integer self-test */
struct self_test bigint_test __self_test = {
	.name = "bigint",
	.exec = bigint_test_exec,
};
