/*
 * Copyright (C) 2000-2005 SWsoft. All rights reserved.
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Trolltech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <signal.h>

#include "list.h"
#include "vzctl.h"
#include "fs.h"
#include "tmpl.h"
#include "logger.h"
#include "vzerror.h"
#include "util.h"
#include "script.h"
#include "lock.h"

#define BACKUP		0
#define DESTR		1
int destroydir(char *dir);

/* Renames (to "*.destroyed" if action == MOVE) or removes config,
 * (if action == DESTR)
 * Also, appropriate mount/umount scripts are linked.
 */
static int move_config(int veid, int action)
{
	char conf[PATH_LEN];
	char newconf[PATH_LEN];

	snprintf(conf, sizeof(conf), VPS_CONF_DIR "%d.conf", veid);
	snprintf(newconf, sizeof(newconf), "%s." DESTR_PREFIX, conf);
	action == BACKUP ? rename(conf, newconf) : unlink(newconf);

	snprintf(conf, sizeof(conf), VPS_CONF_DIR "%d." MOUNT_PREFIX, veid);
	snprintf(newconf, sizeof(newconf), "%s." DESTR_PREFIX, conf);
	action == BACKUP ? rename(conf, newconf) : unlink(newconf);
	
	snprintf(conf, sizeof(conf), VPS_CONF_DIR "%d." UMOUNT_PREFIX, veid);
	snprintf(newconf, sizeof(newconf), "%s." DESTR_PREFIX, conf);
	action == BACKUP ? rename(conf, newconf) : unlink(newconf);

	return 0;
}

const char *get_ostmpl(tmpl_param *tmpl)
{

        if (tmpl->ostmpl != NULL)
                return tmpl->ostmpl;
	else if (tmpl->dist != NULL)
		return tmpl->dist;
        return NULL;
}

static int fs_create(fs_param *fs, tmpl_param *tmpl)
{
	char tarball[PATH_LEN];
	char tmp_dir[PATH_LEN];
	char buf[PATH_LEN];
	int ret, fd;
	char *arg[2];
	char *env[4];
	
	snprintf(tarball, sizeof(tarball), "%s/cache/%s.tar.gz",
		fs->tmpl, tmpl->ostmpl);

	if (!stat_file(tarball)) {
		logger(0, 0, "Cached os template %s not found",	tarball);
		return VZ_PKGSET_NOT_FOUND;
	}
	if (make_dir(fs->private, 0))
		return VZ_CANT_CREATE_DIR;
	/* Lock VPS area */
	if ((fd = open(fs->private, O_WRONLY | O_CREAT | O_TRUNC, 0644)) < 0) {
		logger(0, errno, "Unable to create %s", fs->private);
		return VZ_CANT_CREATE_DIR;
	}
	close(fd);
	snprintf(tmp_dir, sizeof(tmp_dir), "%s.tmp", fs->private);
	if (stat_file(tmp_dir)) {
		logger(0, 0, "Warning: Temp dir %s already exists, deleting",
			tmp_dir);
		if ((ret = destroydir(tmp_dir)))
			goto err;	
	}
	if (make_dir(tmp_dir, 1)) {
		logger(0, errno, "Unable to create directory %s", tmp_dir);
		ret = VZ_FS_NEW_VE_PRVT;
		goto err;
	}
	arg[0] = VPS_CREATE;
	arg[1] = NULL;
	snprintf(buf, sizeof(buf), "PRIVATE_TEMPLATE=%s", tarball);
	env[0] = strdup(buf);
	snprintf(buf, sizeof(buf), "VE_PRVT=%s", tmp_dir);
	env[1] = strdup(buf);
	env[2] = strdup(ENV_PATH);
	env[3] = NULL;

	if ((ret = run_script(VPS_CREATE, arg, env, 0))) {
		destroydir(tmp_dir);
		goto err;
	}
	/* Unlock VPS area */
	unlink(fs->private);
	if (rename(tmp_dir, fs->private)) {
		logger(0, errno, "Can't rename %s to %s",
			tmp_dir, fs->private);
		ret = VZ_CANT_CREATE_DIR;
	}
err:
	unlink(fs->private);
	free_arg(env);

	return ret;
}

int vps_create(vps_handler *h, envid_t veid, fs_param *fs, tmpl_param *tmpl)
{
	int ret;

	if (check_var(fs->tmpl, "TEMPLATE is not set"))
		return VZ_VE_TMPL_NOTSET;
	if (check_var(fs->private, "VE_PRIVATE is not set"))
		return VZ_VE_PRIVATE_NOTSET;
	if (check_var(fs->root, "VE_ROOT is not set"))
		return VZ_VE_ROOT_NOTSET;
	if (check_var(tmpl->ostmpl, "OS template is not specified"))
		return VZ_VE_PKGSET_NOTSET;
	if (stat_file(fs->private)) {
		logger(0, 0, "Private area already exists in %s", fs->private);
		return VZ_FS_PRVT_AREA_EXIST;
	}
	logger(0, 0, "Creating VPS private area: %s", fs->private);
	if ((ret = fs_create(fs, tmpl)))
		return ret;
	move_config(veid, DESTR);
	logger(0, 0, "VPS private area was created");

	return ret;
}

int vps_postcreate(envid_t veid, fs_param *fs, tmpl_param *tmpl)
{
	char buf[STR_SIZE];
	dist_actions actions;
	char *dist_name;
	char *arg[2];
	char *env[3];
	int ret;
 
	if (check_var(fs->root, "VE_ROOT is not set"))
                return VZ_VE_ROOT_NOTSET;
	dist_name = get_dist_name(tmpl);
	if ((ret = read_dist_actions(dist_name, DIST_DIR, &actions)))
		return ret;
	if (dist_name != NULL)
		free(dist_name);
	if (actions.post_create == NULL) {
		ret = 0;
		goto err;
	}
	ret = fsmount(veid, fs, NULL);
	if (ret)
		goto err;
	arg[0] = actions.post_create;
	arg[1] = NULL;
	snprintf(buf, sizeof(buf), "VE_ROOT=%s", fs->root);
	env[0] = buf;
	env[1] = ENV_PATH;
	env[2] = NULL;
	logger(0, 0, "Performing postcreate actions");
	ret = run_script(actions.post_create, arg, env, 0);
	fsumount(veid, fs->root);
err:
	free_dist_actions(&actions);
	return ret;
}

static int destroy(envid_t veid, char *dir)
{
        int ret;

	logger(0, 0, "Destroying VPS private area: %s", dir);

	if (!quota_ctl(veid, QUOTA_STAT)) {
		if ((ret = quota_off(veid, 0)))
			if ((ret = quota_off(veid, 1)))
				return ret;
	}
	quota_ctl(veid, QUOTA_DROP);	
	if ((ret = destroydir(dir)))
		return ret;
	logger(0, 0, "VPS private area was destroyed");

        return 0;
}

static char *get_destroy_root(char *dir)
{
	struct stat st;
	int id, len;
	char *p, *prev;
	char tmp[STR_SIZE];

	if (stat(dir, &st) < 0)
		return NULL;
	id = st.st_dev;
	p = dir + strlen(dir) - 1;
	prev = p;
	while (p > dir) {
		while (p > dir && (*p == '/' || *p == '.')) p--;
		while (p > dir && *p != '/') p--;
		if (p <= dir)
			break;
		len = p - dir + 1;
		strncpy(tmp, dir, len);
		tmp[len] = 0;
		if (stat(tmp, &st) < 0)
			return NULL;
		if (id != st.st_dev)
			break;
		prev = p;
	}
	len = prev - dir;
	if (len) {
		strncpy(tmp, dir, len);
		tmp[len] = 0;
		return strdup(tmp);
	}
	return NULL;
}

char *maketmpdir(const char *dir)
{
        char buf[STR_SIZE];
        char *tmp;
        char *tmp_dir;
	int len;

        snprintf(buf, sizeof(buf), "%s/XXXXXXX", dir);
        if (!(tmp = mkdtemp(buf)))
                return NULL;
	len = strlen(dir);
        tmp_dir = (char *)malloc(strlen(tmp) - len);
        strcpy(tmp_dir, tmp + len + 1);

        return tmp_dir;
}

static void _destroydir(char *root)
{
	char buf[STR_SIZE];
        struct stat st;
        struct dirent *ep;
        DIR *dp;
	int del;

	do {
		if (!(dp = opendir(root)))
			return;
		del = 0;
		while ((ep = readdir(dp))) {
			if (!strcmp(ep->d_name, ".") ||
				!strcmp(ep->d_name, ".."))
			{
				continue;
			}
			snprintf(buf, sizeof(buf), "%s/%s", root, ep->d_name);
			if (stat(buf, &st)) 
				continue;
                	if (!S_ISDIR(st.st_mode))
				continue;
			snprintf(buf, sizeof(buf), "rm -rf %s/%s",
				root, ep->d_name);
			system(buf);
			del = 1;
		}
		closedir(dp);
	} while(del);
}

int destroydir(char *dir)
{
	char buf[STR_SIZE];
	char tmp[STR_SIZE];
	char *root;
	char *tmp_nm;
	int fd_lock, pid;
	struct sigaction act, actold;
	int ret = 0;

	if (!stat_file(dir))
		return 0;
	root = get_destroy_root(dir);
	if (root == NULL) {
		logger(0, 0, "Unable to get root for %s", dir);
		return -1;
	}
	snprintf(tmp, sizeof(buf), "%s/tmp", root);
	free(root);
	if (!stat_file(tmp)) {
		if (mkdir(tmp, 0755)) {
			logger(0, errno, "Can't create tmp dir %s", tmp);
			return VZ_FS_DEL_PRVT;
		}
	}
	/* First move to del */
	if ((tmp_nm = maketmpdir(tmp)) == NULL)	{
		logger(0, 0, "Unable to generate temporary name in %s",
			tmp);
		return VZ_FS_DEL_PRVT;
	}
	snprintf(buf, sizeof(tmp), "%s/%s", tmp, tmp_nm);
	free(tmp_nm);
	if (rename(dir, buf)) {
		logger(0, errno, "Can't move %s -> %s", dir, buf);
		rmdir(buf);
		return VZ_FS_DEL_PRVT;
	}
	snprintf(buf, sizeof(buf), "%s/rm.lck", tmp);
	if ((fd_lock = _lock(buf, 0)) == -2) {
		/* Already locked */
		_unlock(fd_lock, NULL);
		return 0;
	} else if (fd_lock == -1)
		return VZ_FS_DEL_PRVT;
	
	sigaction(SIGCHLD, NULL, &actold);
	sigemptyset(&act.sa_mask);
	act.sa_handler = SIG_IGN;
	act.sa_flags = SA_NOCLDSTOP;
	sigaction(SIGCHLD, &act, NULL);

	if (!(pid = fork())) {
		int fd;

		setsid();
		fd = open("/dev/null", O_WRONLY);
		if (fd != -1) {
			close(0);
			close(1);
			close(2);
			dup2(fd, 1);
			dup2(fd, 2);
		}
		_destroydir(tmp);
		_unlock(fd_lock, buf);
		exit(0);
	} else if (pid < 0)  {
		logger(0, errno, "destroydir: Unable to fork");
		ret = VZ_RESOURCE_ERROR;
	}
	sleep(1);
	sigaction(SIGCHLD, &actold, NULL);
	return ret;
}

int vps_destroy(vps_handler *h, envid_t veid, fs_param *fs)
{
	int ret;
	
	if (check_var(fs->private, "VE_PRIVATE is not set"))
		return VZ_VE_PRIVATE_NOTSET;
	if (check_var(fs->root, "VE_ROOT is not set"))
		return VZ_VE_ROOT_NOTSET;
	if (vps_is_mounted(fs->root)) {
		logger(0, 0, "VPS is currently mounted (umount first)");
                return VZ_FS_MOUNTED;
	}
	if ((ret = destroy(veid, fs->private)))
		return ret;
	move_config(veid, BACKUP);
	rmdir(fs->root);

	return 0;
}
