<?php

namespace Doctrine\Tests\ORM\Functional\Ticket;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\Type;

class DDC3785Test extends \Doctrine\Tests\OrmFunctionalTestCase
{
    protected function setUp() : void
    {
        parent::setUp();

        Type::addType('ddc3785_asset_id', DDC3785_AssetIdType::class);

        try {
            $this->_schemaTool->createSchema(
                [
                    $this->_em->getClassMetadata(DDC3785_Asset::class),
                    $this->_em->getClassMetadata(DDC3785_AssetId::class),
                    $this->_em->getClassMetadata(DDC3785_Attribute::class)
                ]
            );
        } catch(\Exception $e) {
        }
    }

    /**
     * @group DDC-3785
     */
    public function testOwningValueObjectIdIsCorrectlyTransformedWhenRemovingOrphanedChildEntities()
    {
        $id = new DDC3785_AssetId('919609ba-57d9-4a13-be1d-d202521e858a');

        $attributes = [
            $attribute1 = new DDC3785_Attribute('foo1', 'bar1'),
            $attribute2 = new DDC3785_Attribute('foo2', 'bar2')
        ];

        $this->_em->persist($asset = new DDC3785_Asset($id, $attributes));
        $this->_em->flush();

        $asset->getAttributes()
              ->removeElement($attribute1);

        $idToBeRemoved = $attribute1->id;

        $this->_em->persist($asset);
        $this->_em->flush();

        self::assertNull($this->_em->find(DDC3785_Attribute::class, $idToBeRemoved));
        self::assertNotNull($this->_em->find(DDC3785_Attribute::class, $attribute2->id));
    }
}

/**
 * @Entity
 * @Table(name="asset")
 */
class DDC3785_Asset
{
    /**
     * @Id @GeneratedValue(strategy="NONE") @Column(type="ddc3785_asset_id")
     */
    private $id;

    /**
     * @ManyToMany(targetEntity="DDC3785_Attribute", cascade={"persist"}, orphanRemoval=true)
     * @JoinTable(name="asset_attributes",
     *      joinColumns={@JoinColumn(name="asset_id", referencedColumnName="id")},
     *      inverseJoinColumns={@JoinColumn(name="attribute_id", referencedColumnName="id")}
     *      )
     **/
    private $attributes;

    public function __construct(DDC3785_AssetId $id, $attributes = [])
    {
        $this->id = $id;
        $this->attributes = new ArrayCollection();

        foreach ($attributes as $attribute) {
            $this->attributes->add($attribute);
        }
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAttributes()
    {
        return $this->attributes;
    }
}

/**
 * @Entity
 * @Table(name="attribute")
 */
class DDC3785_Attribute
{
    /**
     * @Id @Column(type="integer")
     * @GeneratedValue
     */
    public $id;

    /** @Column(type = "string") */
    private $name;

    /** @Column(type = "string") */
    private $value;

    public function __construct($name, $value)
    {
        $this->name = $name;
        $this->value = $value;
    }
}

/** @Embeddable */
class DDC3785_AssetId
{
    /** @Column(type = "guid") */
    private $id;

    public function __construct($id)
    {
        $this->id = $id;
    }

    public function __toString()
    {
        return $this->id;
    }
}

class DDC3785_AssetIdType extends Type
{
    /**
     * {@inheritdoc}
     */
    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform)
    {
        return $platform->getGuidTypeDeclarationSQL($fieldDeclaration);
    }

    /**
     * {@inheritdoc}
     */
    public function convertToDatabaseValue($value, AbstractPlatform $platform)
    {
        return (string)$value;
    }

    /**
     * {@inheritdoc}
     */
    public function convertToPHPValue($value, AbstractPlatform $platform)
    {
        return new DDC3785_AssetId($value);
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return 'ddc3785_asset_id';
    }
}
